/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.configuration.layers;

import static com.mulesoft.connectivity.rest.commons.api.interception.CompositeHttpResponseInterceptor.ExecutionStrategy.ONLY_FIRST;
import static javax.lang.model.element.Modifier.ABSTRACT;
import static javax.lang.model.element.Modifier.PUBLIC;

import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import com.mulesoft.connectivity.rest.commons.api.interception.CompositeHttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.interception.HttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.interception.expression.ExpressionHttpResponseInterceptor;
import com.mulesoft.connectivity.rest.commons.api.interception.expression.PreconditionHttpResponseInterceptorDelegate;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorInterceptor;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class SdkConfigBaseLayer extends AbstractSdkConfigLayer {

  private static final String GET_RESPONSE_INTERCEPTORS = "getResponseInterceptors";

  private final TypeName superclass;

  public SdkConfigBaseLayer(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, String javaClassName,
                            String packageName, TypeName superclass, RestSdkRunConfiguration runConfiguration) {
    super(outputDir, connectorModel, sdkConnector, javaClassName, packageName, runConfiguration);
    this.superclass = superclass;
  }

  public String getJavaClassName() {
    return super.getJavaClassName() + BASE_CLASSNAME_SUFFIX;
  }

  public String getPackage() {
    return super.getPackage() + BASE_PACKAGE_SUFFIX;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateConfigClass();
  }

  private void generateConfigClass() throws TemplatingException {
    TypeSpec.Builder configClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .addModifiers(ABSTRACT)
            .superclass(superclass);

    if (connectorModel.getInterceptors() != null && connectorModel.getInterceptors().size() > 0) {
      addResponseInterceptors(configClassBuilder);
    }

    JavaFile.Builder javaFileBuilder = getJavaFileBuilderForClass(configClassBuilder.build(), getPackage());
    writeJavaFile(javaFileBuilder.build());
  }

  private CodeBlock.Builder generateInterceptorsBlocks(CodeBlock.Builder methodBody, List<ConnectorInterceptor> interceptors,
                                                       String name, String matcher) {

    String listName = "httpResponseInterceptors" + name;
    methodBody.addStatement("$1T<$2T> " + listName + " = new $3T<>()",
                            List.class,
                            HttpResponseInterceptor.class,
                            ArrayList.class);

    for (ConnectorInterceptor connectorInterceptor : interceptors) {

      methodBody.add(listName + ".add($1T.builder()\n",
                     ExpressionHttpResponseInterceptor.class);


      if (connectorInterceptor.getExpression() != null) {
        methodBody.add(".matchExpression(\"" + connectorInterceptor.getExpression() + "\")\n");
      }
      if (connectorInterceptor.getReasonPhrase() != null) {
        methodBody.add(".reasonPhraseExpression(\"" + connectorInterceptor.getReasonPhrase() + "\")\n");
      }
      if (connectorInterceptor.getStatusCode() != null) {
        methodBody.add(".statusCodeExpression(\"" + connectorInterceptor.getStatusCode() + "\")\n");
      }
      if (connectorInterceptor.getBody() != null) {
        methodBody.add(".bodyExpression(\"" + connectorInterceptor.getBody() + "\")\n");
      }
      if (connectorInterceptor.getHeaders() != null) {
        methodBody.add(".headersExpression(\"" + connectorInterceptor.getHeaders() + "\")\n");
      }
      methodBody.add(".defaultResponseMediaType(getDefaultResponseMediaType())\n");
      methodBody.addStatement(".build())");
    }

    methodBody.addStatement(
                            "$1T " + name + " = new $1T(" + matcher + ", new $2T(httpResponseInterceptors" + name
                                + ","
                                + "$3T.ONLY_FIRST))",
                            PreconditionHttpResponseInterceptorDelegate.class,
                            CompositeHttpResponseInterceptor.class,
                            ONLY_FIRST.getClass());
    return methodBody;
  }

  private CodeBlock generateInterceptors() {

    CodeBlock.Builder methodBody = CodeBlock.builder();

    methodBody.addStatement("$1T.Matcher isRestSdkErrorPrecondition = "
        + "(httpRequest, httpResponse) -> $2T.getErrorByCode(httpResponse.getStatusCode()).isPresent()",
                            PreconditionHttpResponseInterceptorDelegate.class,
                            RestError.class);
    methodBody.addStatement("$1T.Matcher isRestSdkNotErrorPrecondition = "
        + "(httpRequest, httpResponse) -> !$2T.getErrorByCode(httpResponse.getStatusCode()).isPresent()",
                            PreconditionHttpResponseInterceptorDelegate.class,
                            RestError.class);

    methodBody.addStatement("$1T<$2T> httpResponseInterceptorsFinal = new $3T<>()",
                            List.class,
                            HttpResponseInterceptor.class,
                            ArrayList.class);

    List<ConnectorInterceptor> interceptorsOnError =
        connectorModel.getInterceptors().stream().filter(x -> x.isOnError()).collect(Collectors.toList());
    List<ConnectorInterceptor> interceptorsOnSuccess =
        connectorModel.getInterceptors().stream().filter(x -> !x.isOnError()).collect(Collectors.toList());

    if (interceptorsOnError != null && interceptorsOnError.size() > 0) {
      generateInterceptorsBlocks(methodBody, interceptorsOnError, "onErrorInterceptors", "isRestSdkErrorPrecondition");
      methodBody.addStatement("httpResponseInterceptorsFinal.add(onErrorInterceptors)");
    }

    if (interceptorsOnSuccess != null && interceptorsOnSuccess.size() > 0) {
      generateInterceptorsBlocks(methodBody, interceptorsOnSuccess, "onSuccessInterceptors",
                                 "isRestSdkNotErrorPrecondition");
      methodBody.addStatement("httpResponseInterceptorsFinal.add(onSuccessInterceptors)");
    }

    methodBody.addStatement("return new $1T(httpResponseInterceptorsFinal, $2T.ONLY_FIRST)",
                            CompositeHttpResponseInterceptor.class,
                            ONLY_FIRST.getClass());

    return methodBody.build();
  }

  private void addResponseInterceptors(TypeSpec.Builder configClassBuilder) {


    MethodSpec getInterceptors = MethodSpec.methodBuilder(GET_RESPONSE_INTERCEPTORS)
        .returns(TypeName.get(HttpResponseInterceptor.class))
        .addModifiers(PUBLIC)
        .addAnnotation(Override.class)
        .addCode(generateInterceptors())
        .build();

    configClassBuilder.addMethod(getInterceptors);
  }
}
