/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter;

import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.MuleAnnotationsUtils.generateContentAnnotation;

import org.mule.runtime.api.metadata.TypedValue;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata.SdkInputMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.FileGenerationUtil;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;

import java.io.InputStream;
import java.nio.file.Path;
import java.util.List;
import java.util.stream.Collectors;

public class SdkAuxiliarParameter extends SdkParameter {

  private SdkInputMetadataResolver inputMetadataResolver = null;
  private final List<SdkField> sdkFields;

  private boolean generateContentAnnotation;

  public SdkAuxiliarParameter(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                              String containerInternalName, String parentJavaName, Parameter parameter,
                              JavaTemplateEntity parentElement, RestSdkRunConfiguration runConfiguration,
                              boolean generateContentAnnotation)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, parentJavaName, parameter, parentElement, generateContentAnnotation,
          runConfiguration);
    this.generateContentAnnotation = generateContentAnnotation;

    final TypeDefinition typeDefinition = parameter.getTypeDefinition();
    if (typeDefinition.getTypeSchema() != null) {
      this.inputMetadataResolver =
          new SdkInputMetadataResolver(outputDir, connectorModel, sdkConnector, containerInternalName, typeDefinition,
                                       FileGenerationUtil.SchemaNameType.INPUT, parameter.getInternalName(), runConfiguration);
    }

    sdkFields = parameter.getFields()
        .stream()
        .map(field -> new SdkField(outputDir, connectorModel, sdkConnector, field, parentElement, runConfiguration))
        .collect(Collectors.toList());
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    super.applyTemplates();
    if (inputMetadataResolver != null) {
      inputMetadataResolver.applyTemplates();
    }
    for (SdkField sdkField : sdkFields) {
      sdkField.applyTemplates();
    }
  }

  @Override
  protected void configureParameterSpecBuilder(ParameterSpec.Builder paramSpecBuilder) {
    super.configureParameterSpecBuilder(paramSpecBuilder);

    if (inputMetadataResolver != null && inputMetadataResolver.getRequiresMetadataResolver()) {
      paramSpecBuilder
          .addAnnotation(generateMetadataResolverAnnotation());
      if (generateContentAnnotation) {
        paramSpecBuilder.addAnnotation(generateContentAnnotation());
      }

      sdkFields
          .forEach(sdkField -> paramSpecBuilder.addAnnotation(getValueProviderAnnotation(sdkField, parameter.getInternalName())));
    }
  }

  private AnnotationSpec generateMetadataResolverAnnotation() {
    ClassName className = ClassName.get(inputMetadataResolver.getPackage(), inputMetadataResolver.getClassName());
    return com.mulesoft.connectivity.rest.sdk.templating.sdk.util.MuleAnnotationsUtils.getTypeResolverAnnotation(className);
  }

  @Override
  protected void configureFieldSpecBuilder(FieldSpec.Builder fieldSpecBuilder) {
    super.configureFieldSpecBuilder(fieldSpecBuilder);

    if (inputMetadataResolver != null && inputMetadataResolver.getRequiresMetadataResolver()) {
      fieldSpecBuilder
          .addAnnotation(generateMetadataResolverAnnotation());
      if (generateContentAnnotation) {
        fieldSpecBuilder.addAnnotation(generateContentAnnotation());
      }

      sdkFields
          .forEach(sdkField -> fieldSpecBuilder.addAnnotation(getValueProviderAnnotation(sdkField, parameter.getInternalName())));
    }
  }

  @Override
  public TypeName getTypeName() {
    return inputMetadataResolver != null ? ParameterizedTypeName.get(TypedValue.class, InputStream.class) : super.getTypeName();
  }
}
