/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser.selectionsFromBinding;
import static java.util.Arrays.asList;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverDeclaration;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;

import java.nio.file.Path;
import java.util.Map;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public abstract class AbstractSdkResolverProviderReference extends AbstractSdkResolverProvider {

  public static final String RESOLVER_NAME_FIELD = "RESOLVER_NAME";

  private final ResolverReference<?> reference;
  private final SdkResolverDefinition<?> sdkResolverDefinition;
  private final ResolverDeclaration<?> declaration;
  private TypeName superclass;
  private final boolean useExtensionsApi;

  public AbstractSdkResolverProviderReference(Path outputDir,
                                              ConnectorModel connectorModel,
                                              ResolverReference<?> reference,
                                              SdkResolverDefinition<?> sdkResolverDefinition,
                                              RestSdkRunConfiguration runConfiguration,
                                              boolean useExtensionsApi) {
    super(outputDir, connectorModel, runConfiguration);
    this.reference = reference;
    this.sdkResolverDefinition = sdkResolverDefinition;
    this.declaration = reference.getDeclaration();
    this.useExtensionsApi = useExtensionsApi;
  }

  /**
   * Generates the 'build' method of the resolver reference.
   */
  protected void addBuildMethod(TypeSpec.Builder classBuilder) {
    sdkResolverDefinition.addBuildMethod(classBuilder, generateBuildMethodBody());
  }

  private CodeBlock generateBuildMethodBody() {
    CodeBlock.Builder builder = CodeBlock.builder();

    builder.add("builder.reference(referenceBuilder -> referenceBuilder.declaration($1L)", RESOLVER_NAME_FIELD);

    for (Argument arg : reference.getArguments()) {
      builder.add(".argument($1S, argumentBuilder -> argumentBuilder.value(valueBuilder -> valueBuilder.expression($2S)))",
                  arg.getName(),
                  arg.getValue().getValue());
    }
    builder.add(")");

    return builder.build();
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    String declarationKey = declaration.getName() + (useExtensionsApi ? "extensions-api" : EMPTY);

    if (getAllParents().containsKey(declarationKey)) {
      this.superclass = getAllParents().get(declarationKey);
    } else {
      AbstractSdkResolverProviderReferenceParent parent = getNewParent(useExtensionsApi);
      parent.applyTemplates();
      this.superclass = parent.getTypeName();
      getAllParents().put(declarationKey, superclass);
    }

    super.applyTemplates();
  }

  protected abstract Map<String, TypeName> getAllParents();

  protected abstract AbstractSdkResolverProviderReferenceParent getNewParent(boolean useExtensionsApi);

  @Override
  protected TypeName getSuperClass() {
    return superclass;
  }

  @Override
  protected boolean isBoundParameter(SdkParameter sdkParameter) {
    return reference.getArguments().stream()
        .anyMatch(x -> {
          final String[] paramAccessor =
              selectionsFromBinding(x.getValue().getValue(), sdkParameter.getParameterType().getBinding());
          return asList(paramAccessor).contains(sdkParameter.getExternalName());
        });
  }

  @Override
  protected void buildClass(TypeSpec.Builder classBuilder) {
    addParameterFieldsIfNeeded(classBuilder);
    addBuildMethod(classBuilder);
    addConfigureEvaluationContextMethod(classBuilder);
  }
}
