/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.operation;

import com.mulesoft.connectivity.rest.commons.internal.util.StreamUtils;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.MethodSpec;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static javax.lang.model.element.Modifier.PROTECTED;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.util.Reference;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;

import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import org.apache.commons.lang3.tuple.Pair;

public abstract class AbstractSdkPaginationOperation extends AbstractSdkOperation implements SdkPaginationStrategy {

  private final Pagination pagination;
  private static final String BASE_REQUEST_BUILDER_MAIN_METHOD_NAME = "RequestBuilderMain";
  private static final String BASE_PAGING_PROVIDER_MAIN_METHOD_NAME = "PagingProviderMain";
  private static final String PARAMETER_REQUEST_FACTORY = "requestFactory";

  private static final String PARAMETERS_BINDINGS_REF_NAME = "parameterBindingsRef";
  private static final String CUSTOM_PARAMETERS_BINDINGS_REF_NAME = "customParameterBindingsRef";

  public AbstractSdkPaginationOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                        ConnectorOperation operation, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, operation, runConfiguration);
    this.pagination = operation.getPagination();
  }

  public Pagination getPagination() {
    return pagination;
  }


  protected abstract List<ParameterSpec> getPagingParametersAsParameterSpecs() throws TemplatingException;

  @Override
  protected void generateOperationMethodsFlow(TypeSpec.Builder operationClassBuilder) throws TemplatingException {
    operationClassBuilder.addMethod(generateBaseMainMethod());
    operationClassBuilder.addMethod(generateRequestBuilderMainMethod());
    operationClassBuilder.addMethod(generatePagingProviderMainMethod());
  }

  protected void addResponseBindings(CodeBlock.Builder block) {
    if (operation.getResponseBindings() != null && !operation.getResponseBindings().get().isEmpty()) {
      block.add(".withResponseBinding(getResponseBindings())");
    }
    block.add(";");
  }

  @Override
  public CodeBlock getPagingMethodOperation() throws TemplatingException {
    CodeBlock.Builder block = CodeBlock.builder();
    block.add(
              "return new $T($S, $L, $L, getExpressionLanguage(), $S, "
                  + " resolveDefaultResponseMediaType(config), $L, $L, overrides.getResponseTimeoutAsMillis())",
              getPagingProviderClass(),
              pagination.getPaginationParameter(),
              getPagingParameter().getJavaName(),
              PARAMETER_REQUEST_FACTORY,
              getPayloadExpression(),
              PARAMETER_BINDINGS_NAME,
              CUSTOM_PARAMETER_BINDINGS_NAME);
    addResponseBindings(block);
    return block.build();
  }

  protected SdkParameter getPagingParameter() throws TemplatingException {
    return allQueryParameters.stream()
        .filter(x -> x.getExternalName().equalsIgnoreCase(pagination.getPaginationParameter()))
        .findFirst()
        .orElseThrow(() -> new TemplatingException("Could not get paging parameter, this is a bug."));
  }

  @Override
  public TypeName generateMethodReturn() {
    return ParameterizedTypeName.get(ClassName.get(PagingProvider.class),
                                     TypeName.get(RestConnection.class),
                                     ParameterizedTypeName.get(TypedValue.class, String.class));
  }

  @Override
  protected MessageOutputType getMessageOutputType() {
    ParameterizedTypeName parameterizedTypeName = ParameterizedTypeName.get(List.class, String.class);
    TypeName attribute = TypeName.get(Void.class);
    return new MessageOutputType(parameterizedTypeName, attribute);
  }

  @Override
  protected AbstractSdkOperation.ParametersBuilder generateBaseMainMethodParameters() {
    return AbstractSdkOperation
        .builder()
        .configurationParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .customParameterBinding()
        .contentParameter(generateContentParameters(false))
        .requestParameter()
        .configurationOverridesParameter();
  }

  @Override
  protected Pair<List<ParameterSpec>, CodeBlock.Builder> generateOperationImplementationMethodParameters() {
    return AbstractSdkOperation
        .builder()
        .withAnnotations()
        .configurationParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .contentParameter(generateContentParameters(true))
        .requestParameter()
        .configurationOverridesParameter()
        .parametersSpecList();
  }


  @Override
  public CodeBlock generateOperationMethodBaseMainBody() throws TemplatingException {
    CodeBlock.Builder paginationBody = CodeBlock.builder();

    paginationBody.add(generateAuxParameterBindings());
    paginationBody.add(toCursorProviderVariables());

    paginationBody.addStatement("$T<$T,$T> $L = connection -> $L",
                                Function.class,
                                RestConnection.class,
                                RestRequestBuilder.class,
                                PARAMETER_REQUEST_FACTORY,
                                generateRequestBuilderMainCall());

    paginationBody.add(generatePagingProviderCall());

    return paginationBody.build();
  }

  private CodeBlock toCursorProviderVariables() {
    CodeBlock.Builder builder = CodeBlock.builder();

    builder.addStatement("$1T<$2T> $3L = new $1T<>($4T.resolveCursorProvider($5L))", Reference.class, Map.class,
                         PARAMETERS_BINDINGS_REF_NAME,
                         StreamUtils.class, PARAMETER_BINDINGS_NAME);

    builder.addStatement("$1T<$2T> $3L = new $1T<>($4T.resolveCursorProvider($5L))", Reference.class, Map.class,
                         CUSTOM_PARAMETERS_BINDINGS_REF_NAME,
                         StreamUtils.class, CUSTOM_PARAMETER_BINDINGS_NAME);

    return builder.build();
  }

  @Override
  protected List<String> generateOperationMainCallParameters() {
    return AbstractSdkOperation.builder()
        .configurationParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .customParameterBinding()
        .contentParameter(generateContentParameters(false))
        .requestParameter()
        .configurationOverridesParameter()
        .parametersSpecList().getLeft().stream().map(x -> x.name).collect(toList());
  }

  @Override
  protected MethodSpec generateOperationBaseMethod() throws TemplatingException {

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(getJavaMethodName())
        .addModifiers(PROTECTED);

    Pair<List<ParameterSpec>, CodeBlock.Builder> parameters = AbstractSdkOperation
        .builder()
        .configurationParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .contentParameter(generateContentParameters(false))
        .requestParameter()
        .configurationOverridesParameter()
        .parametersSpecList();

    methodBuilder.addParameters(parameters.getLeft());

    TypeName returnType = generateMethodReturn();

    methodBuilder.addStatement("$1T<$2T,$3T> $4L = new $5T<>()", Map.class, String.class, Object.class,
                               CUSTOM_PARAMETER_BINDINGS_NAME,
                               HashMap.class);

    methodBuilder.addCode(generateOperationMainCall(returnType));

    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  private String getRequestBuilderMainName() {
    return getJavaMethodName() + BASE_REQUEST_BUILDER_MAIN_METHOD_NAME;
  }

  private String getPagingProviderMainName() {
    return getJavaMethodName() + BASE_PAGING_PROVIDER_MAIN_METHOD_NAME;
  }


  private List<ParameterSpec> getHttpParamsParameterSpecs() {

    Pair<List<ParameterSpec>, CodeBlock.Builder> parameters = AbstractSdkOperation
        .builder()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParametersNoPaginated())
        .headerParameters(allHeaders)
        .parametersSpecList();

    return parameters.getLeft();
  }


  private CodeBlock generateRequestBuilderMainCall() {
    CodeBlock.Builder requestBuilderCall = CodeBlock.builder();

    AbstractSdkOperation.ParametersBuilder parameters = AbstractSdkOperation.builder();
    parameters.requestParameter();
    parameters.configurationOverridesParameter();
    parameters.connectionParameter();
    parameters.configurationParameter();
    parameters.parameterBinding();
    parameters.customParameterBinding();

    List<String> parametersParsed = parameters.parametersSpecList().getLeft().stream()
        .map(x -> x.name)
        .map(name -> {
          if (name.equals(PARAMETER_BINDINGS_NAME)) {
            return PARAMETERS_BINDINGS_REF_NAME + ".get()";
          } else if (name.equals(CUSTOM_PARAMETER_BINDINGS_NAME)) {
            return CUSTOM_PARAMETERS_BINDINGS_REF_NAME + ".get()";
          } else {
            return name;
          }
        })
        .collect(toList());
    parametersParsed.addAll(getHttpParamsParameterSpecs().stream().map(param -> param.name).collect(toList()));

    String parametersSpec = "$1L(" + parametersParsed.stream().collect(joining(", ")) + ")";

    requestBuilderCall.add(parametersSpec, getRequestBuilderMainName());

    return requestBuilderCall.build();
  }

  private CodeBlock generatePagingProviderCall() throws TemplatingException {
    CodeBlock.Builder requestBuilderCall = CodeBlock.builder();

    List<String> parameters = new ArrayList() {

      {
        add(PARAMETER_REQUEST_FACTORY);
        add("config");
        add("overrides");
        add(PARAMETER_BINDINGS_NAME);
        add(CUSTOM_PARAMETER_BINDINGS_NAME);
      }
    };
    parameters.addAll(getPagingParametersAsParameterSpecs().stream().map(param -> param.name).collect(toList()));
    String parametersSpec = "return $L(" + parameters.stream().collect(joining(", ")) + ")";

    requestBuilderCall.addStatement(parametersSpec, getPagingProviderMainName());

    return requestBuilderCall.build();
  }

  private void generateRequestBuilderMainMethodSignature(MethodSpec.Builder methodSpec) {
    AbstractSdkOperation.ParametersBuilder parameters = AbstractSdkOperation
        .builder();
    parameters.requestParameter();
    parameters.configurationOverridesParameter();
    parameters.connectionParameter();
    parameters.configurationParameter();
    parameters.parameterBinding();
    parameters.customParameterBinding();

    parameters.uriParameters(allUriParameters);
    parameters.queryParameters(generateQueryParametersNoPaginated());
    parameters.headerParameters(allHeaders);

    methodSpec.addParameters(parameters.parametersSpecList().getLeft());
  }

  public MethodSpec generateRequestBuilderMainMethod() {

    MethodSpec.Builder methodSpecBuilder =
        MethodSpec.methodBuilder(getRequestBuilderMainName())
            .addModifiers(PROTECTED);

    methodSpecBuilder.returns(RestRequestBuilder.class);

    generateRequestBuilderMainMethodSignature(methodSpecBuilder);

    methodSpecBuilder.addCode(generateRestRequestBuilder());

    methodSpecBuilder.addStatement("return builder");

    return methodSpecBuilder.build();
  }

  private void generatePagingProviderMainMethodSignature(MethodSpec.Builder methodSpec) throws TemplatingException {
    ParameterSpec.Builder parameterBindingsSpec;
    parameterBindingsSpec = ParameterSpec.builder(ParameterizedTypeName.get(ClassName.get(Function.class),
                                                                            TypeName.get(RestConnection.class),
                                                                            TypeName.get(RestRequestBuilder.class)),
                                                  PARAMETER_REQUEST_FACTORY);

    List<ParameterSpec> parameterSpecs = new ArrayList<>();
    parameterSpecs.add(parameterBindingsSpec.build());


    AbstractSdkOperation.ParametersBuilder parameters = AbstractSdkOperation
        .builder()
        .configurationParameter()
        .configurationOverridesParameter()
        .parameterBinding()
        .customParameterBinding();

    parameterSpecs.addAll(parameters.parametersSpecList().getLeft());
    parameterSpecs.addAll(getPagingParametersAsParameterSpecs());

    methodSpec.addParameters(parameterSpecs);
  }

  public MethodSpec generatePagingProviderMainMethod() throws TemplatingException {

    MethodSpec.Builder methodSpecBuilder =
        MethodSpec.methodBuilder(getPagingProviderMainName())
            .addModifiers(PROTECTED);

    generatePagingProviderMainMethodSignature(methodSpecBuilder);

    methodSpecBuilder.addCode(getPagingMethodOperation());

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodSpecBuilder.returns(returnType);
    }
    return methodSpecBuilder.build();
  }

  protected List<SdkParameter> generateQueryParametersNoPaginated() {
    if (pagination.getPaginationParameter() != null) {
      return allQueryParameters.stream()
          .filter(x -> !pagination.getPaginationParameter().equals(x.getExternalName()))
          .collect(toList());
    } else {
      return allQueryParameters.stream()
          .collect(toList());
    }
  }

  @Override
  public String getPayloadExpression() {
    return this.pagination.getPaginationResponseExpression();
  }

  @Override
  protected boolean requiresMediaTypeAnnotation() {
    return false;
  }

}
