/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.sampledata;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity.getConstantStringField;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.SdkResolverUtil.getBindingMethod;
import static javax.lang.model.element.Modifier.PROTECTED;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.commons.api.datasense.sampledata.PagedRestSampleDataProvider;
import com.mulesoft.connectivity.rest.commons.api.datasense.sampledata.RestSampleDataProvider;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.sampledata.SampleDataResolverDeclarationBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.builder.sampledata.SampleDataResolverExpressionBuilder;
import com.mulesoft.connectivity.rest.commons.internal.model.sampledata.SampleDataResolverDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.sampledata.SampleDataDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.SdkResolverDefinition;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public class SdkSampleDataDefinition implements SdkResolverDefinition<SampleDataDefinition> {

  public static final String PATH_TEMPLATE_FIELD = "PATH";
  public static final String SAMPLE_DATA_SUFFIX = "SampleDataProvider";

  private final SampleDataDefinition definition;
  private final boolean isTrigger;
  private final boolean isPaged;

  public SdkSampleDataDefinition(ResolverExpression<SampleDataDefinition> resolverExpression, boolean isTrigger,
                                 boolean isPaged) {
    this.definition = buildDefinition(resolverExpression);
    this.isTrigger = isTrigger;
    this.isPaged = isPaged;
  }

  private SampleDataDefinition buildDefinition(ResolverExpression<SampleDataDefinition> resolverExpression) {

    if (resolverExpression instanceof SampleDataDefinition) {
      return (SampleDataDefinition) resolverExpression;
    } else if (resolverExpression instanceof ResolverReference) {
      ResolverReference<SampleDataDefinition> valueProviderReference =
          (ResolverReference<SampleDataDefinition>) resolverExpression;

      return valueProviderReference.getDeclaration().getResolverDefinition();
    }

    throw new IllegalArgumentException("Invalid resolverExpression. This is a bug.");
  }

  @Override
  public CodeBlock.Builder getResolverExpressionBuilder() {
    CodeBlock.Builder codeBlockBuilder = CodeBlock.builder();
    codeBlockBuilder.add(
                         "builder.definition(definitionBuilder -> definitionBuilder.result(httpRequestBuilder -> httpRequestBuilder.path($1L).method($2S)",
                         PATH_TEMPLATE_FIELD, definition.getResult().getMethod());
    if (isNotBlank(definition.getResult().getOutputMediaType())) {
      codeBlockBuilder.add(".outputMediaType($1S)", definition.getResult().getOutputMediaType());
    }

    HttpRequestBinding bindings = definition.getResult().getHttpRequestBinding();
    if (bindings != null) {
      codeBlockBuilder.add(".bindings(bindingBuilder -> bindingBuilder");
      bindings.getHeader().forEach(binding -> codeBlockBuilder.add(getBindingCode(binding, HEADER)));
      bindings.getQueryParameter().forEach(binding -> codeBlockBuilder.add(getBindingCode(binding, QUERY)));
      bindings.getUriParameter().forEach(binding -> codeBlockBuilder.add(getBindingCode(binding, URI)));
      codeBlockBuilder.add(")");
    }
    codeBlockBuilder.add(")");

    if (needsTransformation()) {
      codeBlockBuilder.add(".transform(t -> t.scriptExpression(expressionBuilder -> expressionBuilder.expression($S)))",
                           getTransformation());
    }

    codeBlockBuilder.add(")");

    return codeBlockBuilder;
  }

  private boolean needsTransformation() {
    return isNotBlank(getTransformation());
  }

  private String getTransformation() {
    return definition.getTransformation() == null ? null : definition.getTransformation().getExpression();
  }

  private static CodeBlock getBindingCode(Argument binding, ParameterType parameterType) {
    CodeBlock.Builder bindingCodeBuilder = CodeBlock.builder();

    bindingCodeBuilder
        .add(".$1L($2S, argumentBuilder -> argumentBuilder.value(expressionBuilder -> expressionBuilder.expression($3S)))",
             getBindingMethod(parameterType),
             binding.getName(),
             binding.getValue().getValue());

    return bindingCodeBuilder.build();
  }

  @Override
  public void addClassConstants(TypeSpec.Builder resolverClassBuilder) {
    resolverClassBuilder
        .addField(getConstantStringField(PATH_TEMPLATE_FIELD, definition.getResult().getPath()));
  }

  @Override
  public void addBuildMethod(TypeSpec.Builder classBuilder, CodeBlock methodBody) {
    MethodSpec.Builder buildMethodBuilder =
        MethodSpec.methodBuilder("build")
            .addModifiers(PROTECTED)
            .addParameter(TypeName.get(SampleDataResolverExpressionBuilder.class), "builder")
            .addAnnotation(Override.class);

    buildMethodBuilder.addStatement(methodBody);

    classBuilder.addMethod(buildMethodBuilder.build());
  }

  @Override
  public String getEvaluationContextKind() {
    return isTrigger ? EVALUATION_CONTEXT_KIND_TRIGGER : EVALUATION_CONTEXT_KIND_OPERATION;
  }

  @Override
  public SampleDataDefinition getDefinition() {
    return definition;
  }

  @Override
  public Class<?> getSdkResolverDefinitionClass() {
    return SampleDataResolverDefinition.class;
  }

  @Override
  public Class<?> getSdkResolverDeclarationBuilderClass() {
    return SampleDataResolverDeclarationBuilder.class;
  }

  public Class<?> getSuperclass() {
    return isPaged ? PagedRestSampleDataProvider.class : RestSampleDataProvider.class;
  }
}
