/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.configuration.layers;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme.SecuritySchemeType.CUSTOM_AUTHENTICATION;
import static java.util.stream.Collectors.toList;
import static javax.lang.model.element.Modifier.PUBLIC;
import static org.mule.sdk.api.meta.JavaVersion.JAVA_8;
import static org.mule.sdk.api.meta.JavaVersion.JAVA_11;
import static org.mule.sdk.api.meta.JavaVersion.JAVA_17;

import org.mule.runtime.extension.api.annotation.Configuration;
import org.mule.runtime.extension.api.annotation.Operations;
import org.mule.runtime.extension.api.annotation.Sources;
import org.mule.runtime.extension.api.annotation.connectivity.ConnectionProviders;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.ConnectorSecurityScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.CustomAuthenticationScheme;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.connection.AbstractSdkConnectionProvider;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.connection.SdkConnectionProvider;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.connection.SdkCustomConnectionProvider;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.model.SdkHttpProxyConfig;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.operation.AbstractSdkOperation;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.operation.SdkOperationFactory;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.SdkNativeTrigger;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.SdkTrigger;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.SdkTriggerTemplate;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import org.apache.commons.lang3.StringUtils;
import org.mule.sdk.api.annotation.JavaVersionSupport;

public class SdkConfigImplementationLayer extends AbstractSdkConfigLayer {

  private final SdkHttpProxyConfig httpProxyConfig;
  private final List<AbstractSdkConnectionProvider> connectionProviders;
  private final List<AbstractSdkOperation> operations;
  private final List<SdkTriggerTemplate> triggers;

  private final TypeName superclass;

  public SdkConfigImplementationLayer(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                      String javaClassName, String packageName, TypeName superclass,
                                      RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, javaClassName, packageName, runConfiguration);
    this.superclass = superclass;

    this.httpProxyConfig = new SdkHttpProxyConfig(outputDir, connectorModel, runConfiguration);
    this.connectionProviders = buildConnectionProviders(outputDir, connectorModel, sdkConnector);

    this.operations = new ArrayList<>();
    for (ConnectorOperation operation : connectorModel.getOperations()) {
      this.operations.add(SdkOperationFactory.create(outputDir, connectorModel, sdkConnector, operation, runConfiguration));
    }

    this.triggers = new ArrayList<>();
    for (Trigger triggerModel : connectorModel.getTriggers()) {
      SdkTriggerTemplate trigger =
          (StringUtils.isNotBlank(triggerModel.getFqn()))
              ? new SdkNativeTrigger(outputDir, connectorModel, triggerModel, runConfiguration)
              : new SdkTrigger(outputDir, connectorModel, sdkConnector, triggerModel, runConfiguration);

      triggers.add(trigger);
    }
  }

  private List<AbstractSdkConnectionProvider> buildConnectionProviders(Path outputDir, ConnectorModel connectorModel,
                                                                       SdkConnector sdkConnector)
      throws TemplatingException {
    List<ConnectorSecurityScheme> securitySchemeModels = connectorModel.getOperations().stream()
        .map(ConnectorOperation::getSecuritySchemes)
        .flatMap(List::stream)
        .distinct()
        .collect(toList());

    connectorModel.getTriggers().stream().map(Trigger::getOperation)
        .filter(Objects::nonNull)
        .map(ConnectorOperation::getSecuritySchemes)
        .flatMap(List::stream)
        .filter(securityScheme -> !securitySchemeModels.contains(securityScheme))
        .distinct()
        .forEach(securitySchemeModels::add);

    final List<AbstractSdkConnectionProvider> list = new ArrayList<>();
    for (ConnectorSecurityScheme scheme : securitySchemeModels) {
      if (scheme.getSchemeType().equals(CUSTOM_AUTHENTICATION)) {
        if (((CustomAuthenticationScheme) scheme).getFqn().isPresent()) {
          list.add(
                   new SdkCustomConnectionProvider(outputDir, connectorModel, (CustomAuthenticationScheme) scheme,
                                                   runConfiguration));
        }
      } else {
        list.add(new SdkConnectionProvider(outputDir, connectorModel, sdkConnector, scheme, httpProxyConfig, runConfiguration));
      }
    }
    return list;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    generateConfigClass();

    if (!connectionProviders.isEmpty()) {
      httpProxyConfig.applyTemplates();
    }

    for (AbstractSdkConnectionProvider connectionProvider : connectionProviders) {
      connectionProvider.applyTemplates();
    }

    for (AbstractSdkOperation operation : operations) {
      operation.applyTemplates();
    }

    for (SdkTriggerTemplate trigger : triggers) {
      trigger.applyTemplates();
    }
  }

  private void generateConfigClass() throws TemplatingException {
    TypeSpec.Builder configClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(superclass)
            .addAnnotation(getConfigurationAnnotation())
            .addAnnotation(getOperationsAnnotation())
            .addAnnotation(getConnectionProvidersAnnotation())
            .addAnnotation(getJavaVersionSupportAnnotation());

    if (!triggers.isEmpty()) {
      configClassBuilder.addAnnotation(getSourcesAnnotation());
    }

    writeClassToFile(configClassBuilder.build(), getPackage());
  }

  private AnnotationSpec getSourcesAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < triggers.size(); i++) {
      SdkTriggerTemplate trigger = triggers.get(i);
      codeBlock.add("$T.class", trigger.getTypeNameForConfig());
      if (i < triggers.size() - 1) {
        codeBlock.add(",");
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder sourcesAnnotationBuilder =
        AnnotationSpec
            .builder(Sources.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return sourcesAnnotationBuilder.build();
  }

  private AnnotationSpec getOperationsAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < operations.size(); i++) {
      AbstractSdkOperation operation = operations.get(i);
      if (!operation.getOperation().isPrivateOperation()) {
        codeBlock.add("$T.class", ClassName.get(operation.getPackage(), operation.getJavaClassName()));
        if (i < operations.size() - 1) {
          codeBlock.add(",");
        }
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder operationsAnnotationBuilder =
        AnnotationSpec
            .builder(Operations.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return operationsAnnotationBuilder.build();
  }

  private AnnotationSpec getConnectionProvidersAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{");

    for (int i = 0; i < connectionProviders.size(); i++) {
      AbstractSdkConnectionProvider connectionProvider = connectionProviders.get(i);
      codeBlock.add("$T.class", connectionProvider.getTypeNameForConfig());
      if (i < connectionProviders.size() - 1) {
        codeBlock.add(",");
      }
    }

    codeBlock.add("}");

    AnnotationSpec.Builder connectionProvidersAnnotationBuilder =
        AnnotationSpec
            .builder(ConnectionProviders.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return connectionProvidersAnnotationBuilder.build();
  }

  private AnnotationSpec getJavaVersionSupportAnnotation() {
    CodeBlock.Builder codeBlock = CodeBlock.builder();

    codeBlock.add("{$T.$L,$T.$L,$T.$L}",
                  org.mule.sdk.api.meta.JavaVersion.class, JAVA_8,
                  org.mule.sdk.api.meta.JavaVersion.class, JAVA_11,
                  org.mule.sdk.api.meta.JavaVersion.class, JAVA_17);

    AnnotationSpec.Builder connectionProvidersAnnotationBuilder =
        AnnotationSpec
            .builder(JavaVersionSupport.class)
            .addMember(VALUE_MEMBER, codeBlock.build());

    return connectionProvidersAnnotationBuilder.build();
  }

  private AnnotationSpec getConfigurationAnnotation() {
    AnnotationSpec.Builder extensionAnnotationBuilder =
        AnnotationSpec
            .builder(Configuration.class);

    return extensionAnnotationBuilder.build();
  }

  public List<AbstractSdkOperation> getOperations() {
    return this.operations;
  }
}
