/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.operation;

import static com.mulesoft.connectivity.rest.commons.internal.util.DwUtils.isExpression;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser.getOutputMediaType;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.AUXILIAR;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.BODY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition.simplePrimitiveType;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.abbreviateText;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaLowerCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.SdkResolverUtil.getActingParameterJavaName;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.sampledata.SdkSampleDataFactory.getSdkSampleDataResolver;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkBodyLevelUtils.transformBodyPrefix;
import static java.lang.String.format;
import static java.util.Arrays.stream;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static javax.lang.model.element.Modifier.ABSTRACT;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA_TYPE;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.runtime.http.api.HttpHeaders.Values.BOUNDARY;

import com.mulesoft.connectivity.rest.commons.api.error.RestError;
import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Ignore;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.metadata.OutputResolver;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.exception.ModuleException;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.sdk.api.annotation.binding.Binding;
import org.mule.sdk.api.annotation.data.sample.SampleData;

import com.mulesoft.connectivity.rest.commons.api.binding.HttpRequestBinding;
import com.mulesoft.connectivity.rest.commons.api.binding.HttpResponseBinding;
import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.error.RequestErrorTypeProvider;
import com.mulesoft.connectivity.rest.commons.api.operation.BaseRestOperation;
import com.mulesoft.connectivity.rest.commons.api.operation.ConfigurationOverrides;
import com.mulesoft.connectivity.rest.commons.api.operation.RequestParameters;
import com.mulesoft.connectivity.rest.commons.internal.RestConstants;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.StreamUtils;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.body.Body;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.body.Field;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestDataExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dw.DataWeaveExpressionParser;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.operation.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.AuxiliarParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.sampledata.SampleDataDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata.SdkOutputMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata.SdkPagingMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.munit.SdkMtfOperationTest;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkAuxiliarParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkContent;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkField;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkPaginationParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.SdkResolverTemplate;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.MuleAnnotationsUtils;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.valueprovider.SdkValueProvidable;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.io.InputStream;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

public abstract class AbstractSdkOperation extends JavaTemplateEntity implements SdkValueProvidable {

  public static final String PARAM_DOC_NAME_DESCRIPTION = "@param $L $L\n";

  public static final String CONTENT_TYPE_HEADER_NAME = "content-type";
  public static final String ACCEPT_HEADER_NAME = "accept";

  public static final String ADD_URI_PARAM_METHOD_NAME = "addUriParam";
  public static final String ADD_QUERY_PARAM_METHOD_NAME = "addQueryParam";
  public static final String ADD_HEADER_METHOD_NAME = "addHeader";

  private static final String REQUEST_PARAMETERS_GROUP_NAME = "REQUEST_PARAMETERS_GROUP_NAME";
  private static final String CONNECTOR_OVERRIDES = "CONNECTOR_OVERRIDES";

  private static final String GET_REQUEST_BODY_DATA_TYPE_METHOD = "getRequestBodyMediaType";
  private static final String GET_RESPONSE_BODY_DATA_TYPE_METHOD = "getResponseBodyMediaType";

  private static final String QUERY_PARAM_FORMAT_FIELD = "QUERY_PARAM_FORMAT";
  public static final String OPERATION_PATH_FIELD = "OPERATION_PATH";

  private static final String BASE_MAIN_METHOD_NAME = "Main";

  protected static final String PARAMETER_BINDINGS_NAME = "parameterBindings";
  protected static final String CUSTOM_PARAMETER_BINDINGS_NAME = "customParameterBindings";

  public static final String OPERATION_CLASSNAME_SUFFIX = "Operation";

  protected final ConnectorOperation operation;
  private final SdkConnector sdkConnector;
  protected final List<SdkParameter> allUriParameters;
  protected final List<SdkParameter> allQueryParameters;
  protected final List<SdkParameter> allHeaders;
  protected final List<SdkParameter> auxParameters;
  protected final List<SdkField> allBodyFields;
  protected final SdkContent content;
  private final Optional<String> muleOutputResolver;
  protected final SdkOutputMetadataResolver outputMetadataResolver;

  protected final SdkResolverTemplate sampleDataProvider;

  private final SdkMtfOperationTest sdkMtfOperationTest;

  public abstract TypeName generateMethodReturn();

  public abstract CodeBlock generateOperationMethodBaseMainBody() throws TemplatingException;

  public AbstractSdkOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                              ConnectorOperation operation, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);

    this.operation = operation;
    this.sdkConnector = sdkConnector;

    this.allUriParameters = buildSdkParameters(outputDir, connectorModel, sdkConnector, operation.getUriParameters(), false);
    this.allQueryParameters = buildSdkParameters(outputDir, connectorModel, sdkConnector, operation.getQueryParameters(), true);
    this.allHeaders = buildSdkParameters(outputDir, connectorModel, sdkConnector, operation.getHeaders(), false);
    this.auxParameters = new ArrayList<>();
    final List<AuxiliarParameter> parameters = operation.getParameters().orElse(null);
    if (parameters != null) {
      this.auxParameters
          .addAll(buildSdkParameters(operation, parameters));
    }
    this.allBodyFields = buildSdkFields(outputDir, connectorModel, sdkConnector, operation.getBody());

    this.content = buildContent(outputDir, connectorModel, sdkConnector, operation);

    this.muleOutputResolver = ofNullable(StringUtils.trim(operation.getMuleOutputResolver()));
    this.outputMetadataResolver = (muleOutputResolver.isPresent()) ? null : buildOutputMetadataResolver(sdkConnector);

    this.sdkMtfOperationTest = new SdkMtfOperationTest(operation, outputDir);
    this.sampleDataProvider = buildSdkSampleDataProvider(operation);
  }

  private SdkResolverTemplate buildSdkSampleDataProvider(ConnectorOperation operation) {
    return getSdkSampleDataResolver(operation, connectorModel, sdkConnector, getJavaClassName(), getAllParameters(), outputDir,
                                    runConfiguration);
  }

  private List<SdkField> buildSdkFields(Path outputDir,
                                        ConnectorModel connectorModel,
                                        SdkConnector sdkConnector,
                                        Body body) {
    final List<SdkField> list = new ArrayList<>();
    if (body == null) {
      return list;
    }

    for (Field field : body.getFields()) {
      list.add(new SdkField(outputDir, connectorModel, sdkConnector, field, this, runConfiguration));
    }

    return list;
  }

  private List<SdkParameter> buildSdkParameters(Path outputDir,
                                                ConnectorModel connectorModel,
                                                SdkConnector sdkConnector,
                                                List<Parameter> connectorParameters,
                                                boolean checkPaginationParameters) {
    final List<SdkParameter> list = new ArrayList<>();

    for (Parameter parameter : connectorParameters) {
      if (checkPaginationParameters && isQueryParamDefinedInPagination(parameter.getExternalName())) {
        Pagination pagination = getPagination();

        list.add(new SdkPaginationParameter(outputDir, connectorModel, sdkConnector, getJavaClassName(),
                                            parameter, this, runConfiguration, pagination));
      } else {
        list.add(new SdkParameter(outputDir, connectorModel, sdkConnector, getJavaClassName(), parameter, this,
                                  runConfiguration));
      }
    }

    return list;
  }

  private List<SdkParameter> buildSdkParameters(ConnectorOperation operation,
                                                List<AuxiliarParameter> connectorAuxParameters)
      throws TemplatingException {
    final List<SdkParameter> list = new ArrayList<>();
    if (connectorAuxParameters != null) {
      for (AuxiliarParameter parameter : connectorAuxParameters) {
        list.add(buildSdkParameter(parameter, operation));
      }
    }
    return list;
  }

  private SdkParameter buildSdkParameter(AuxiliarParameter auxiliarParameter, ConnectorOperation operation)
      throws TemplatingException {
    boolean generateContentAnnotation = ofNullable(auxiliarParameter.getMuleContent()).orElse(true);
    TypeDefinition typeDefinition = auxiliarParameter.getTypeDefinition();
    if (typeDefinition == null && auxiliarParameter.getMuleTypeResolver() == null) {
      typeDefinition = getTypeDefinition(auxiliarParameter.getType());
      // since primitive types are defaulted not to add @Content annotation, we set this here, BTW it can be set by the user
      generateContentAnnotation = ofNullable(auxiliarParameter.getMuleContent()).orElse(false);
    }
    // TODO RSDK-620: [Tech Debt] Refactor parameter hierarchy, the rest sdk generatar should not instantiate classes of other
    com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter parameter =
        new com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter(
                                                                                           auxiliarParameter.getDisplayName(),
                                                                                           auxiliarParameter.getExternalName(),
                                                                                           auxiliarParameter.getExternalName(),
                                                                                           AUXILIAR,
                                                                                           typeDefinition,
                                                                                           auxiliarParameter.getMuleAlias(),
                                                                                           auxiliarParameter.getDescription(),
                                                                                           auxiliarParameter.isRequired() == null
                                                                                               || auxiliarParameter.isRequired(),
                                                                                           null,
                                                                                           false,
                                                                                           auxiliarParameter.getValueProvider(),
                                                                                           auxiliarParameter
                                                                                               .getMuleMetadataKeyId(),
                                                                                           auxiliarParameter
                                                                                               .getMuleTypeResolver(),
                                                                                           auxiliarParameter
                                                                                               .getMuleContent(),
                                                                                           auxiliarParameter.getFields());
    return new SdkAuxiliarParameter(outputDir, connectorModel, sdkConnector, operation.getInternalName(), getJavaClassName(),
                                    parameter, this,
                                    runConfiguration, generateContentAnnotation);
  }

  private TypeDefinition getTypeDefinition(ParameterDataType parameterDataType) {
    PrimitiveTypeDefinition.PrimitiveType primitiveType;

    switch (parameterDataType) {
      case NUMBER:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.NUMBER;
        break;
      case INTEGER:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.INTEGER;
        break;
      case LONG:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.LONG;
        break;
      case STRING:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.STRING;
        break;
      case BOOLEAN:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.BOOLEAN;
        break;
      case LOCAL_DATE_TIME:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.LOCAL_DATE_TIME;
        break;
      case ZONED_DATE_TIME:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.ZONED_DATE_TIME;
        break;
      case BINARY:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.BINARY;
        break;
      default:
        throw new IllegalArgumentException("Parameter type not supported. This is a bug.");
    }

    return simplePrimitiveType(primitiveType);
  }

  public ConnectorOperation getOperation() {
    return operation;
  }

  private SdkOutputMetadataResolver buildOutputMetadataResolver(SdkConnector sdkConnector) throws TemplatingException {
    if (operation.getOutputMetadata() == null) {
      return null;
    }

    if (operation.hasPagination()) {
      return new SdkPagingMetadataResolver(outputDir,
                                           connectorModel,
                                           sdkConnector,
                                           operation.getInternalName(),
                                           operation.getOutputMetadata(),
                                           runConfiguration);
    } else {
      return new SdkOutputMetadataResolver(outputDir,
                                           connectorModel,
                                           sdkConnector,
                                           operation.getInternalName(),
                                           operation.getOutputMetadata(),
                                           EMPTY,
                                           runConfiguration);
    }
  }

  protected CodeBlock toCursorProviderMapCodeBlock(String... localVariables) {
    CodeBlock.Builder builder = CodeBlock.builder();
    for (String localVariable : localVariables) {
      builder.addStatement("$1L = $2T.resolveCursorProvider($1L)", localVariable, StreamUtils.class);
    }
    return builder.build();
  }

  protected SdkContent buildContent(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                    ConnectorOperation operation)
      throws TemplatingException {

    if (operation.isRequestBodyBound()) {
      return null;
    }
    return operation.getInputMetadata() != null
        ? new SdkContent(outputDir, connectorModel, sdkConnector, operation, this, runConfiguration)
        : null;
  }

  @Override
  public String getJavaClassName() {
    return getJavaUpperCamelNameFromXml(operation.getInternalName()) + OPERATION_CLASSNAME_SUFFIX;
  }

  public String getJavaBaseClassName() {
    return getJavaClassName() + BASE_CLASSNAME_SUFFIX;
  }

  public String getJavaInterceptorClassName() {
    return getJavaClassName() + REFINEMENT_CLASSNAME_SUFFIX;
  }

  protected String getJavaMethodName() {
    return getJavaLowerCamelNameFromXml(operation.getInternalName());
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + ".internal.operation";
  }

  public String getInterceptorPackage() {
    return getPackage() + REFINEMENT_PACKAGE_SUFFIX;
  }

  public String getBasePackage() {
    return getPackage() + BASE_PACKAGE_SUFFIX;
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (content != null) {
      content.applyTemplates();
    }

    if (hasOutput() && outputMetadataResolver != null) {
      outputMetadataResolver.applyTemplates();
    }

    for (SdkParameter sdkParameter : allUriParameters) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : allQueryParameters) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : allHeaders) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : auxParameters) {
      sdkParameter.applyTemplates();
    }

    if (!runConfiguration.regenerateMode()) {
      sdkMtfOperationTest.applyTemplates();
    }
    if (sampleDataProvider != null) {
      sampleDataProvider.applyTemplates();
    }

    generateOperationImplementationLayer();
    generateOperationInterceptorLayer();
    generateOperationBaseLayer();
  }

  private MethodSpec generateGetRequestBodyDataTypeMethod() {
    return generateGetBodyDataTypeMethod(GET_REQUEST_BODY_DATA_TYPE_METHOD, getRequestBodyMediaType());
  }

  private String getRequestBodyMediaType() {
    return getBodyMediaType(getOperation().getInputMetadata());
  }

  private MethodSpec generateGetResponseBodyDataTypeMethod() {
    return generateGetBodyDataTypeMethod(GET_RESPONSE_BODY_DATA_TYPE_METHOD, getResponseBodyMediaType());
  }

  private String getResponseBodyMediaType() {
    return getBodyMediaType(getOperation().getOutputMetadata());
  }

  private String getBodyMediaType(TypeDefinition typeDefinition) {
    String result;
    if (typeDefinition != null && typeDefinition.getMediaType() != null) {
      result = typeDefinition.getMediaType().toString();
    } else {
      // Default value
      result = MediaType.APPLICATION_JSON;
    }
    return result;
  }

  private MethodSpec generateGetBodyDataTypeMethod(String name, String bodyMediaType) {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(name)
            .addModifiers(PROTECTED)
            .returns(String.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("return \"$L\"", bodyMediaType);

    return methodBuilder.build();
  }

  protected void generateOperationImplementationLayer() throws TemplatingException {
    TypeSpec.Builder operationClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(ClassName.get(getInterceptorPackage(), getJavaInterceptorClassName()))
            .addJavadoc("Lower part of the Operation. It has the operation declaration with its annotations.")
            .addMethod(generateOperationImplementationMethod());

    defineConstructors(operationClassBuilder);

    configureClassBuilder(operationClassBuilder);
    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getPackage(), operationClassBuilder.build())
        .skipJavaLangImports(true)
        .addStaticImport(RestConstants.class, REQUEST_PARAMETERS_GROUP_NAME, CONNECTOR_OVERRIDES);

    configureJavaFileBuilder(javaFileBuilder);

    writeJavaFile(javaFileBuilder.build());
  }

  protected void generateOperationInterceptorLayer() throws TemplatingException {
    TypeSpec.Builder operationClassBuilder =
        TypeSpec
            .classBuilder(getJavaInterceptorClassName())
            .addModifiers(PUBLIC)
            .addJavadoc("Middle part of the Operation. Can be used by the user to add custom code into the operation.")
            .superclass(ClassName.get(getBasePackage(), getJavaBaseClassName()));

    defineConstructors(operationClassBuilder);

    configureClassBuilder(operationClassBuilder);
    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getInterceptorPackage(), operationClassBuilder.build())
        .skipJavaLangImports(true);

    configureJavaFileBuilder(javaFileBuilder);

    writeJavaFile(javaFileBuilder.build(), true, operation.isRefined());
  }

  protected void generateOperationMethodsFlow(TypeSpec.Builder operationClassBuilder) throws TemplatingException {
    operationClassBuilder.addMethod(generateBaseMainMethod());
  }

  protected void generateOperationBaseLayer() throws TemplatingException {
    MethodSpec operationMethod = generateOperationBaseMethod();

    TypeSpec.Builder operationClassBuilder =
        TypeSpec
            .classBuilder(getJavaBaseClassName())
            .addModifiers(PUBLIC)
            .addModifiers(ABSTRACT)
            .superclass(getSuperclass())
            .addJavadoc("Higher part of the Operation. It has the implementation of the operation.")
            .addMethod(operationMethod);

    defineConstructors(operationClassBuilder);

    operationClassBuilder.addField(generateOperationPathField());
    operationClassBuilder.addField(generateQueryParamFormatField());

    generateOperationMethodsFlow(operationClassBuilder);

    operationClassBuilder.addMethod(generateGetRequestBodyDataTypeMethod());
    operationClassBuilder.addMethod(generateGetResponseBodyDataTypeMethod());
    operationClassBuilder.addMethod(generateRequestBindingsMethod());
    operationClassBuilder.addMethod(generateResponseBindingsMethod());

    configureClassBuilder(operationClassBuilder);
    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getBasePackage(), operationClassBuilder.build())
        .skipJavaLangImports(true);

    configureJavaFileBuilder(javaFileBuilder);

    writeJavaFile(javaFileBuilder.build());
  }

  private void defineConstructors(TypeSpec.Builder operationClassBuilder) {
    stream(getSuperclass().getDeclaredConstructors()).forEach(constructor -> {
      MethodSpec.Builder constructorBuilder = MethodSpec.constructorBuilder().addModifiers(Modifier.PUBLIC);
      StringBuilder methodStatement = new StringBuilder("super(");
      for (int i = 0; i < constructor.getParameterTypes().length; i++) {
        Class parameterType = constructor.getParameterTypes()[i];
        String name = "arg" + i;
        constructorBuilder.addParameter(parameterType, name);
        if (i > 0) {
          methodStatement.append(", ");
        }
        methodStatement.append(name);
      }
      methodStatement.append(")");
      constructorBuilder.addStatement(methodStatement.toString());
      operationClassBuilder.addMethod(constructorBuilder.build());
    });
  }

  protected void configureJavaFileBuilder(JavaFile.Builder javaFileBuilder) {}

  protected void configureClassBuilder(TypeSpec.Builder operationClassBuilder) {}

  protected Class<? extends BaseRestOperation> getSuperclass() {
    return BaseRestOperation.class;
  }

  private FieldSpec generateQueryParamFormatField() {
    return FieldSpec.builder(RestRequestBuilder.ParameterArrayFormat.class,
                             QUERY_PARAM_FORMAT_FIELD,
                             PROTECTED,
                             Modifier.FINAL,
                             Modifier.STATIC)
        .initializer("$T.$L", RestRequestBuilder.ParameterArrayFormat.class, operation.getQueryParamArrayFormat().name())
        .build();
  }

  private FieldSpec generateOperationPathField() {
    return FieldSpec.builder(String.class,
                             OPERATION_PATH_FIELD,
                             PROTECTED,
                             Modifier.FINAL,
                             Modifier.STATIC)
        .initializer("$S", operation.getPath())
        .build();
  }

  private AnnotationSpec generateDescriptionAnnotation() {
    return AnnotationSpec
        .builder(Summary.class)
        .addMember(VALUE_MEMBER, "$S", abbreviateText(operation.getDescription()))
        .build();
  }

  private CodeBlock.Builder generateJavadoc() {
    return CodeBlock.builder()
        .add("\n$L\n", defaultIfEmpty(operation.getDescription(), operation.getDisplayName()))
        .add("\nThis operation makes an HTTP $L request to the $L endpoint", operation.getHttpMethod().name().toUpperCase(),
             operation.getPath())
        .add("\n");
  }

  private void addAnnotations(MethodSpec.Builder methodBuilder) {
    methodBuilder.addAnnotation(generateThrowsAnnotation());
    methodBuilder.addAnnotation(generateDisplayNameAnnotation());

    if (operation.isSidecar()) {
      methodBuilder.addAnnotation(generateSidecarAnnotation());
    }

    if (isNotBlank(operation.getDescription())) {
      methodBuilder.addAnnotation(generateDescriptionAnnotation());
    }

    if (isNotBlank(operation.getAlias())) {
      methodBuilder.addAnnotation(generateAliasAnnotation());
    }

    if (sampleDataProvider != null) {
      methodBuilder.addAnnotation(generateSampleDataAnnotation());
    }

    if (requiresMediaTypeAnnotation()) {
      if (operation.getOutputMetadata() != null) {
        methodBuilder.addAnnotation(generateMediaTypeAnnotation());
      } else {
        methodBuilder.addAnnotation(generateDefaultMediaTypeAnnotation());
      }
    }

    if (hasOutput()) {
      methodBuilder.addAnnotation(getOutputResolverAnnotation());
    }
  }

  protected Pair<List<ParameterSpec>, CodeBlock.Builder> generateOperationImplementationMethodParameters() {
    return AbstractSdkOperation
        .builder()
        .withAnnotations()
        .configurationParameter()
        .connectionParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .contentParameter(generateContentParameters(true))
        .requestParameter()
        .configurationOverridesParameter()
        .callbackParameter(getMessageOutputType())
        .parametersSpecList();
  }

  public MethodSpec generateOperationImplementationMethod() throws TemplatingException {

    CodeBlock.Builder javaDoc = generateJavadoc();

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(getJavaMethodName())
        .addModifiers(PUBLIC);

    Pair<List<ParameterSpec>, CodeBlock.Builder> parameters = generateOperationImplementationMethodParameters();

    methodBuilder.addParameters(parameters.getLeft());
    javaDoc.add(parameters.getRight().build());

    addAnnotations(methodBuilder);

    TypeName returnType = generateMethodReturn();

    methodBuilder.addCode(generateOperationImplementationMethodBody(methodBuilder.parameters, returnType));

    methodBuilder.addJavadoc(javaDoc.build());

    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  protected List<String> generateOperationMainCallParameters() {
    return AbstractSdkOperation.builder()
        .configurationParameter()
        .connectionParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .customParameterBinding()
        .contentParameter(generateContentParameters(false))
        .requestParameter()
        .configurationOverridesParameter()
        .callbackParameter(getMessageOutputType())
        .parametersSpecList().getLeft().stream().map(x -> x.name).collect(toList());
  }

  public CodeBlock generateOperationMainCall(TypeName returnType) {
    final CodeBlock.Builder builder = CodeBlock.builder();
    String returnSentence = EMPTY;
    if (returnType != null) {
      returnSentence = "return";
    }
    List<String> params = generateOperationMainCallParameters();

    builder.addStatement("$L $L($L)",
                         returnSentence,
                         getBaseMainMethodName(),
                         params.stream().collect(joining(", ")));
    return builder.build();
  }

  protected MethodSpec generateOperationBaseMethod() throws TemplatingException {

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(getJavaMethodName())
        .addModifiers(PROTECTED);

    AbstractSdkOperation.ParametersBuilder parameters = AbstractSdkOperation
        .builder()
        .configurationParameter()
        .connectionParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .contentParameter(generateContentParameters(false))
        .requestParameter()
        .configurationOverridesParameter()
        .callbackParameter(getMessageOutputType());

    methodBuilder.addParameters(parameters.parametersSpecList().getLeft());

    TypeName returnType = generateMethodReturn();

    generateTryStatement(methodBuilder, true);

    methodBuilder.addStatement("$1T<$2T,$3T> $4L = new $5T<>()", Map.class, String.class, Object.class,
                               CUSTOM_PARAMETER_BINDINGS_NAME,
                               HashMap.class);

    methodBuilder.addCode(generateOperationMainCall(returnType));

    generateTryStatement(methodBuilder, false);
    generateCatchStatement(ModuleException.class, "callback.error(e);", methodBuilder);
    generateCatchStatement(Throwable.class,
                           "callback.error(new $1T(\"Unknown error\", $2T.CONNECTIVITY, e));",
                           methodBuilder, ModuleException.class, RestError.CONNECTIVITY.getClass());

    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  protected Pair<List<ParameterSpec>, CodeBlock.Builder> generateContentParameters(boolean withAnnotations) {
    if (content != null) {
      ParameterSpec parameterSpec = content.generateContentParameter(withAnnotations);
      return new ImmutablePair<>(new ArrayList<ParameterSpec>() {

        {
          add(parameterSpec);
        }
      }, CodeBlock.builder().add(
                                 PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                                 "the content to use"));
    }
    return null;
  }

  private AnnotationSpec getOutputResolverAnnotation() {
    AnnotationSpec.Builder builder = AnnotationSpec
        .builder(OutputResolver.class);

    if (outputMetadataResolver != null && outputMetadataResolver.getRequiresMetadataResolver()) {
      builder.addMember("output", "$T.class",
                        ClassName.get(outputMetadataResolver.getPackage(),
                                      outputMetadataResolver.getClassName()));
    } else if (muleOutputResolver.isPresent()) {
      ClassName className = MuleAnnotationsUtils.getClassName(muleOutputResolver.get());
      builder.addMember("output", "$T.class", className);
    }
    addOutputAttributesMetadataResolver(builder);
    return builder.build();
  }

  private void addOutputAttributesMetadataResolver(AnnotationSpec.Builder builder) {
    builder.addMember("attributes", "$T.class", sdkConnector.getOutputAttributesResolver());
  }

  private Pagination getPagination() {
    return this.operation.getPagination();
  }

  protected boolean isQueryParamDefinedInPagination(String paramName) {
    Pagination pagination = getPagination();
    if (pagination != null && pagination.getPaginationParameter() != null) {
      return pagination.getPaginationParameter().equalsIgnoreCase(paramName);
    }

    return false;
  }

  private AnnotationSpec generateDisplayNameAnnotation() {
    return AnnotationSpec.builder(DisplayName.class)
        .addMember(VALUE_MEMBER, "$S", operation.getDisplayName())
        .build();
  }

  private AnnotationSpec generateAliasAnnotation() {
    return AnnotationSpec
        .builder(Alias.class)
        .addMember(VALUE_MEMBER, "$S", operation.getAlias())
        .build();
  }

  private AnnotationSpec generateSidecarAnnotation() {
    return AnnotationSpec.builder(Ignore.class)
        .build();
  }

  public CodeBlock generateOperationImplementationMethodBody(List<ParameterSpec> parameters, TypeName typeName)
      throws TemplatingException {
    final CodeBlock.Builder builder = CodeBlock.builder();
    String returnSentence = EMPTY;
    if (typeName != null) {
      returnSentence = "return";
    }
    builder.addStatement("$L super.$L($L)",
                         returnSentence,
                         getJavaMethodName(),
                         parameters.stream().map(param -> param.name).collect(joining(", ")));
    return builder.build();
  }

  protected void generateTryStatement(MethodSpec.Builder methodBody, boolean start) {
    if (start) {
      methodBody.addCode("try {");
    } else {
      methodBody.addCode("}");
    }
  }

  protected void generateCatchStatement(Class exception, String codeBlock, MethodSpec.Builder methodBody, Object... args) {
    methodBody.addCode("$1L ($2T e) {", "catch", exception);
    methodBody.addCode(codeBlock, args);
    methodBody.addCode("}");
  }

  private String getBaseUriString() {
    String baseUriString = "connection.getBaseUri()";

    if (isNotBlank(operation.getAlternativeBaseUri()) && connectorModel.getBaseUri().isMultipleBaseUri()) {
      String multipleBaseUri =
          connectorModel.getBaseUri().getMultipleBaseUriOrDefault(operation.getAlternativeBaseUri());

      baseUriString = "\"" + multipleBaseUri + "\"";
    }
    return baseUriString;
  }

  public CodeBlock generateRestRequestBuilder() {
    CodeBlock.Builder builder = CodeBlock.builder();

    builder.add(generateRestRequestBuilderInitialization(getBaseUriString()));

    builder.add(".setQueryParamFormat($L)", QUERY_PARAM_FORMAT_FIELD);

    if (connectorModel.getInterceptors() != null && connectorModel.getInterceptors().size() > 0) {
      builder.add(".$L($L($L))", "responseInterceptorDescriptor", "getResponseInterceptorDescriptor", "config");
    }

    if (operation.getInputMetadata() != null && operation.getInputMetadata().getMediaType() != null) {
      addContentTypeHeader(builder);
    }

    if (operation.getOutputMetadata() != null && operation.getOutputMetadata().getMediaType() != null) {
      builder.add(".$L($S, $S)",
                  ADD_HEADER_METHOD_NAME,
                  ACCEPT_HEADER_NAME,
                  operation.getOutputMetadata().getMediaType().toString());
    }

    for (SdkParameter uriParameter : allUriParameters) {
      builder.add(generateRequestBuilderParameterCodeBlock(uriParameter, ADD_URI_PARAM_METHOD_NAME));
    }

    for (SdkParameter queryParam : allQueryParameters) {
      if (!isQueryParamDefinedInPagination(queryParam.getExternalName())) {
        builder.add(generateRequestBuilderParameterCodeBlock(queryParam, ADD_QUERY_PARAM_METHOD_NAME));
      }
    }

    for (SdkParameter header : allHeaders) {
      builder.add(generateRequestBuilderParameterCodeBlock(header, ADD_HEADER_METHOD_NAME));
    }

    addSetBodyMethod(builder);

    builder.add(";");
    return builder.build();
  }

  private void addContentTypeHeader(CodeBlock.Builder methodBody) {
    javax.ws.rs.core.MediaType contentType = operation.getInputMetadata().getMediaType();

    if (operation.getInputMetadata().getMediaType().equals(MULTIPART_FORM_DATA_TYPE)) {
      Map<String, String> parameters = new HashMap<>();
      // Default boundary from RSDK
      parameters.put(BOUNDARY, "__rc2_34b212");

      Optional<String> optionalBodyExpression = operation.getRequestBindings().flatMap(
                                                                                       requestBindings -> requestBindings
                                                                                           .stream()
                                                                                           .filter(bindings -> bindings
                                                                                               .getParameterType().equals(BODY))
                                                                                           .filter(bindings -> isExpression(bindings
                                                                                               // TODO: (RSDK-728)
                                                                                               .getExpression().trim()))
                                                                                           .map(bindings -> bindings
                                                                                               .getExpression())
                                                                                           .findFirst());
      if (optionalBodyExpression.isPresent()) {
        String bodyExpression = optionalBodyExpression.get();
        javax.ws.rs.core.MediaType outputMediaType =
            getOutputMediaType(bodyExpression)
                .orElseThrow(() -> new IllegalStateException(
                                                             format("There should be an output directive declaring the mimeType multipart for the body expression for operation ['%s']. This is a bug. Expression: %s",
                                                                    operation.getInternalName(), bodyExpression)));

        if (!operation.getInputMetadata().getMediaType().isCompatible(outputMediaType)) {
          throw new IllegalStateException(format("Operation ['%s'] defines a body expression with an output mimeType '%s' which is not compatible with the input operation mimeType '%s'. This is a bug",
                                                 operation.getInternalName(), outputMediaType,
                                                 operation.getInputMetadata().getMediaType()));
        }

        if (!outputMediaType.getParameters().containsKey(BOUNDARY)) {
          throw new IllegalStateException(format("Operation ['%s'] defines a body expression with an output mimeType '%s' without the boundary parameter. This is a bug",
                                                 operation.getInternalName(), outputMediaType));
        }

        parameters.putAll(outputMediaType.getParameters());
      }
      // Append the boundary parameter to the media type
      contentType = new javax.ws.rs.core.MediaType(contentType.getType(), contentType.getSubtype(), parameters);
    }

    methodBody.add(".$L($S, $S)",
                   ADD_HEADER_METHOD_NAME,
                   CONTENT_TYPE_HEADER_NAME,
                   org.mule.runtime.api.metadata.MediaType.parse(contentType.toString()).toRfcString());
  }

  private CodeBlock generateRestRequestBuilderInitialization(String baseUriString) {
    CodeBlock.Builder builder = CodeBlock.builder();
    AbstractSdkOperation.ParametersBuilder parameters = AbstractSdkOperation.builder();
    parameters.requestParameter();
    parameters.configurationOverridesParameter();
    parameters.connectionParameter();
    parameters.configurationParameter();
    parameters.parameterBinding();
    parameters.customParameterBinding();
    builder.add("$T builder = $L($L, $L, $T.$L, " +
        parameters.parametersSpecList().getLeft().stream().map(x -> x.name).collect(joining(", ")) + ")",
                RestRequestBuilder.class,
                GET_REQUEST_BUILDER_WITH_BINDINGS_METHOD,
                baseUriString,
                OPERATION_PATH_FIELD,
                HttpConstants.Method.class,
                operation.getHttpMethod().name().toUpperCase());

    return builder.build();
  }

  protected void addSetBodyMethod(CodeBlock.Builder methodBody) {
    if (content != null) {
      methodBody.add(".setBody($L, overrides.getStreamingType())", content.getContentParameterJavaName());
    }
  }

  private CodeBlock generateRequestBuilderParameterCodeBlock(SdkParameter parameter, String addSingleValueMethodName) {
    CodeBlock.Builder builder = CodeBlock.builder();
    builder.add(".$L($S, $L)", addSingleValueMethodName, parameter.getExternalName(), getParameterValueStatement(parameter));
    return builder.build();
  }

  protected CodeBlock getParameterValueStatement(SdkParameter parameter) {
    CodeBlock.Builder builder = CodeBlock.builder();

    if (parameter.isArrayType()) {
      CodeBlock getter = parameter.getInnerTypeStringValueGetter("v");
      if (getter.toString().equalsIgnoreCase("v")) {
        builder.add("$L.stream().filter($T::nonNull).collect($T.toList())",
                    parameter.getJavaName(),
                    Objects.class,
                    Collectors.class);
      } else {
        builder.add("$L.stream().filter($T::nonNull).map(v -> $L).collect($T.toList())",
                    parameter.getJavaName(),
                    Objects.class,
                    getter,
                    Collectors.class);
      }
    } else {
      builder.add("$L", parameter.getStringValueGetter());
    }

    return builder.build();
  }

  private AnnotationSpec generateThrowsAnnotation() {
    return AnnotationSpec
        .builder(Throws.class)
        .addMember(VALUE_MEMBER, "$T.class", RequestErrorTypeProvider.class)
        .build();
  }

  private AnnotationSpec generateMediaTypeAnnotation() {
    return AnnotationSpec
        .builder(MediaType.class)
        .addMember(VALUE_MEMBER, "$S", operation.getOutputMetadata().getMediaType())
        .build();
  }

  private AnnotationSpec generateSampleDataAnnotation() {
    AnnotationSpec.Builder annotationBuilder = AnnotationSpec
        .builder(SampleData.class)
        .addMember(VALUE_MEMBER, "$T.class", sampleDataProvider.getTypeName());
    if (operation.getSampleData() instanceof ResolverReference) {
      ResolverReference<SampleDataDefinition> sampleDataDefinitionReferenceResolver =
          (ResolverReference<SampleDataDefinition>) operation.getSampleData();
      addBindingAnnotation(annotationBuilder, sampleDataDefinitionReferenceResolver.getArguments());
    } else if (operation.getSampleData() instanceof SampleDataDefinition) {
      SampleDataDefinition sampleDataDefinition = (SampleDataDefinition) operation.getSampleData();
      HttpRequestDataExpression result = sampleDataDefinition.getResult();
      if (result != null && result.getHttpRequestBinding() != null) {
        List<Argument> arguments = Stream.of(result.getHttpRequestBinding().getHeader(),
                                             result.getHttpRequestBinding().getQueryParameter(),
                                             result.getHttpRequestBinding().getUriParameter())
            .flatMap(argumentList -> argumentList.stream())
            .collect(toList());
        addBindingAnnotation(annotationBuilder, arguments);
      }
    }
    return annotationBuilder.build();
  }

  private void addBindingAnnotation(AnnotationSpec.Builder annotationBuilder, List<Argument> arguments) {
    for (Argument argument : arguments) {
      if (DataWeaveExpressionParser.isBodyBindingUsed(argument))
        annotationBuilder.addMember("bindings", "$L",
                                    AnnotationSpec.builder(Binding.class)
                                        .addMember("actingParameter", "$S", getActingParameterJavaName(argument))
                                        .addMember("extractionExpression", "$S",
                                                   transformBodyPrefix(argument.getValue().getValue(),
                                                                       content.getContentParameterJavaName()))
                                        .build());
    }
  }

  private AnnotationSpec generateDefaultMediaTypeAnnotation() {
    return AnnotationSpec
        .builder(MediaType.class)
        .addMember(VALUE_MEMBER, "$S", operationMethodRequiresBody() ? "application/json" : "text/plain")
        .build();
  }

  /**
   * Utility method to encapsulate how the "return" type (or the output of the current operation's types) is calculated
   */
  protected MessageOutputType getMessageOutputType() {
    if (hasOutput()) {
      return new MessageOutputType(InputStream.class, Object.class);
    } else {
      return new MessageOutputType(Void.class, Void.class);
    }
  }

  protected boolean operationMethodRequiresBody() {
    return operation.getHttpMethod().equals(HTTPMethod.GET)
        || operation.getHttpMethod().equals(HTTPMethod.POST)
        || operation.getHttpMethod().equals(HTTPMethod.PATCH)
        || operation.getHttpMethod().equals(HTTPMethod.OPTIONS);
  }

  protected boolean isVoidOperation() {
    return operation.getVoidOperation() != null && operation.getVoidOperation();
  }

  protected boolean requiresMediaTypeAnnotation() {
    return hasOutput();
  }

  protected boolean hasOutput() {
    return !isVoidOperation() && (outputMetadataResolver != null || operationMethodRequiresBody());
  }

  /**
   * Returns all the parameters of this operation (URI + QUERY + HEADER)
   */
  @Override
  public List<SdkParameter> getAllParameters() {
    final List<SdkParameter> allParameters = new ArrayList<>();
    allParameters.addAll(allUriParameters);
    allParameters.addAll(allQueryParameters);
    allParameters.addAll(allHeaders);
    allParameters.addAll(auxParameters);
    return allParameters;
  }

  public List<SdkField> getAllBodyFields() {
    return allBodyFields;
  }

  public ConnectorOperation getConnectorOperation() {
    return this.operation;
  }

  public static class MessageOutputType {

    private final TypeName outputType;
    private final TypeName attributeOutputType;

    public MessageOutputType(TypeName outputType, TypeName attributeOutputType) {
      this.outputType = outputType;
      this.attributeOutputType = attributeOutputType;
    }

    public MessageOutputType(Class<?> outputType, Class<?> attributeOutputType) {
      this(ClassName.get(outputType), ClassName.get(attributeOutputType));
    }

    public TypeName getOutputType() {
      return outputType;
    }

    public TypeName getAttributeOutputType() {
      return attributeOutputType;
    }
  }

  protected List<SdkParameter> generateQueryParameters() {
    return allQueryParameters;
  }

  // Operation Adapters - IO Transformations
  private static final String GET_REQUEST_BINDINGS_METHOD = "getRequestBindings";
  private static final String GET_RESPONSE_BINDINGS_METHOD = "getResponseBindings";
  private static final String PARAMETER_BINDING_LOCAL_VARIABLE = "bindings";
  private static final String ADD_URI_BINDING_METHOD = "addUriParamBinding";
  private static final String ADD_QUERY_BINDING_METHOD = "addQueryParamBinding";
  private static final String ADD_HEADER_BINDING_METHOD = "addHeaderBinding";
  private static final String ADD_BODY_BINDING_METHOD = "setBody";
  private static final String GET_REQUEST_BUILDER_WITH_BINDINGS_METHOD = "getRequestBuilderWithBindings";

  protected CodeBlock generateAuxParameterBindings() {
    CodeBlock.Builder builder = CodeBlock.builder();

    builder.addStatement("$1T<$2T,$3T> $4L = new $5T<>()", Map.class, String.class, Object.class, PARAMETER_BINDINGS_NAME,
                         HashMap.class);
    for (SdkParameter parameter : auxParameters) {
      builder.addStatement("$L.$L($S, $L)", PARAMETER_BINDINGS_NAME, "put", parameter.getExternalName(),
                           parameter.getJavaName());
    }

    return builder.build();
  }

  private String getBaseMainMethodName() {
    return getJavaMethodName() + BASE_MAIN_METHOD_NAME;
  }

  protected AbstractSdkOperation.ParametersBuilder generateBaseMainMethodParameters() {
    return AbstractSdkOperation
        .builder()
        .configurationParameter()
        .connectionParameter()
        .uriParameters(allUriParameters)
        .queryParameters(generateQueryParameters())
        .headerParameters(allHeaders)
        .auxParameters(auxParameters)
        .customParameterBinding()
        .contentParameter(generateContentParameters(false))
        .requestParameter()
        .configurationOverridesParameter()
        .callbackParameter(getMessageOutputType());
  }

  protected MethodSpec generateBaseMainMethod() throws TemplatingException {
    MethodSpec.Builder methodSpecBuilder =
        MethodSpec.methodBuilder(getBaseMainMethodName())
            .addModifiers(PROTECTED);

    methodSpecBuilder.addParameters(generateBaseMainMethodParameters().parametersSpecList().getLeft());

    methodSpecBuilder.addCode(generateOperationMethodBaseMainBody());

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodSpecBuilder.returns(returnType);
    }

    return methodSpecBuilder.build();
  }

  private MethodSpec generateRequestBindingsMethod() {
    return generateRequestBindingMethod(GET_REQUEST_BINDINGS_METHOD,
                                        HttpRequestBinding.class,
                                        operation.getRequestBindings().orElse(null));
  }

  private MethodSpec generateResponseBindingsMethod() {
    return generateRequestBindingMethod(GET_RESPONSE_BINDINGS_METHOD,
                                        HttpResponseBinding.class,
                                        operation.getResponseBindings().orElse(null));
  }

  private MethodSpec generateRequestBindingMethod(String methodName,
                                                  Class<?> builderClass,
                                                  List<ParameterBinding> parameterBindings) {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(methodName)
            .addModifiers(PROTECTED)
            .returns(builderClass)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("$1T $2L = new $1T()",
                               builderClass,
                               PARAMETER_BINDING_LOCAL_VARIABLE);

    if (parameterBindings != null) {
      for (ParameterBinding binding : parameterBindings) {
        if (!binding.isIgnored() && binding.getExpression() != null) {
          if (binding.getParameterType().equals(BODY)) {
            methodBuilder.addStatement("$L.$L($S)",
                                       PARAMETER_BINDING_LOCAL_VARIABLE,
                                       ADD_BODY_BINDING_METHOD,
                                       // TODO: (RSDK-728) hack to fix org.mule.runtime.api.el.ExpressionExecutionException:
                                       // Unbalanced
                                       // brackets in expression
                                       binding.getExpression().trim());
          } else {
            methodBuilder.addStatement("$L.$L($S, $S)",
                                       PARAMETER_BINDING_LOCAL_VARIABLE,
                                       getParameterBindingAddMethodName(binding),
                                       binding.getName(),
                                       // TODO: (RSDK-728) hack to fix org.mule.runtime.api.el.ExpressionExecutionException:
                                       // Unbalanced
                                       // brackets in expression
                                       binding.getExpression().trim());
          }
        }
      }
    }

    methodBuilder.addStatement("return $1L", PARAMETER_BINDING_LOCAL_VARIABLE);

    return methodBuilder.build();
  }

  private String getParameterBindingAddMethodName(ParameterBinding binding) {
    switch (binding.getParameterType()) {
      case QUERY:
        return ADD_QUERY_BINDING_METHOD;
      case HEADER:
        return ADD_HEADER_BINDING_METHOD;
      case URI:
        return ADD_URI_BINDING_METHOD;
    }

    throw new IllegalArgumentException(format("Parameter type not supported: '%s'. This is a bug.", binding.getParameterType()));
  }

  protected static AbstractSdkOperation.ParametersBuilder builder() {
    return new AbstractSdkOperation.ParametersBuilder();
  }

  public static class ParametersBuilder {

    private boolean withAnnotations;
    private Pair<ParameterSpec, CodeBlock.Builder> configuration;
    private Pair<ParameterSpec, CodeBlock.Builder> connection;
    private Pair<List<ParameterSpec>, CodeBlock.Builder> uriParameters;
    private Pair<List<ParameterSpec>, CodeBlock.Builder> queryParameters;
    private Pair<List<ParameterSpec>, CodeBlock.Builder> headerParameters;
    private Pair<List<ParameterSpec>, CodeBlock.Builder> auxParameters;
    private Pair<ParameterSpec, CodeBlock.Builder> parameterBinding;
    private Pair<List<ParameterSpec>, CodeBlock.Builder> contentParameter;
    private Pair<ParameterSpec, CodeBlock.Builder> requestParameter;
    private Pair<ParameterSpec, CodeBlock.Builder> configurationOverridesParameter;
    private Pair<ParameterSpec, CodeBlock.Builder> callbackParameter;
    private List<ParameterSpec> finalParameters = new ArrayList<>();
    private CodeBlock.Builder javaDoc = CodeBlock.builder();
    private Pair<ParameterSpec, CodeBlock.Builder> customParameterBinding;

    public ParametersBuilder() {
      this.withAnnotations = false;
    }

    public AbstractSdkOperation.ParametersBuilder withAnnotations() {
      this.withAnnotations = true;
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder configurationParameter() {
      ParameterSpec.Builder parameterSpecBuilder = ParameterSpec.builder(RestConfiguration.class, "config");
      if (withAnnotations) {
        parameterSpecBuilder.addAnnotation(Config.class);
      }
      ParameterSpec configParameter = parameterSpecBuilder.build();
      this.configuration = new ImmutablePair<>(configParameter,
                                               CodeBlock.builder().add(PARAM_DOC_NAME_DESCRIPTION, configParameter.name,
                                                                       "the configuration to use"));
      this.finalParameters.add(configParameter);
      this.javaDoc.add(this.configuration.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder connectionParameter() {
      ParameterSpec.Builder parameterSpecBuilder = ParameterSpec.builder(RestConnection.class, "connection");
      if (withAnnotations) {
        parameterSpecBuilder.addAnnotation(Connection.class);
      }
      ParameterSpec connectionParameter = parameterSpecBuilder.build();
      this.connection = new ImmutablePair<>(connectionParameter,
                                            CodeBlock.builder().add(PARAM_DOC_NAME_DESCRIPTION, connectionParameter.name,
                                                                    "the connection to use"));

      this.finalParameters.add(connectionParameter);
      this.javaDoc.add(this.connection.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder uriParameters(List<SdkParameter> allUriParameters) {
      CodeBlock.Builder javaDoc = CodeBlock.builder();
      List<ParameterSpec> parameterSpecs = new ArrayList<>();
      for (SdkParameter sdkParam : allUriParameters) {
        ParameterSpec parameterSpec = sdkParam.generateParameterParameter(withAnnotations).build();
        javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                    defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
        parameterSpecs.add(parameterSpec);
      }

      this.uriParameters = new ImmutablePair<>(parameterSpecs, javaDoc);
      this.finalParameters.addAll(parameterSpecs);
      this.javaDoc.add(this.uriParameters.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder queryParameters(List<SdkParameter> allQueryParameters) {
      CodeBlock.Builder javaDoc = CodeBlock.builder();
      List<ParameterSpec> parameterSpecs = new ArrayList<>();

      for (SdkParameter sdkParam : allQueryParameters) {
        ParameterSpec parameterSpec = sdkParam.generateParameterParameter(withAnnotations).build();
        javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                    defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
        parameterSpecs.add(parameterSpec);
      }

      this.queryParameters = new ImmutablePair<>(parameterSpecs, javaDoc);
      this.finalParameters.addAll(parameterSpecs);
      this.javaDoc.add(this.queryParameters.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder headerParameters(List<SdkParameter> allHeaders) {
      CodeBlock.Builder javaDoc = CodeBlock.builder();
      List<ParameterSpec> parameterSpecs = new ArrayList<>();
      for (SdkParameter sdkParam : allHeaders) {
        ParameterSpec parameterSpec = sdkParam.generateParameterParameter(withAnnotations).build();
        javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                    defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
        parameterSpecs.add(parameterSpec);
      }

      this.headerParameters = new ImmutablePair<>(parameterSpecs, javaDoc);
      this.finalParameters.addAll(parameterSpecs);
      this.javaDoc.add(this.headerParameters.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder auxParameters(List<SdkParameter> auxParameters) {
      CodeBlock.Builder javaDoc = CodeBlock.builder();
      List<ParameterSpec> parameterSpecs = new ArrayList<>();
      for (SdkParameter sdkParam : auxParameters) {
        ParameterSpec parameterSpec = sdkParam.generateParameterParameter(withAnnotations).build();
        javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                    defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
        parameterSpecs.add(parameterSpec);
      }

      this.auxParameters = new ImmutablePair<>(parameterSpecs, javaDoc);
      this.finalParameters.addAll(parameterSpecs);
      this.javaDoc.add(this.auxParameters.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder parameterBinding() {

      ParameterSpec.Builder parameterBindingsSpec;
      parameterBindingsSpec = ParameterSpec.builder(ParameterizedTypeName.get(ClassName.get(Map.class),
                                                                              TypeName.get(String.class),
                                                                              TypeName.get(Object.class)),
                                                    PARAMETER_BINDINGS_NAME);

      ParameterSpec parameterSpec = parameterBindingsSpec.build();

      this.parameterBinding = new ImmutablePair<>(parameterSpec,
                                                  CodeBlock.builder().add(
                                                                          PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                                                                          "Map that contains auxiliary parameters defined in operation"));
      this.finalParameters.add(parameterSpec);
      this.javaDoc.add(this.parameterBinding.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder customParameterBinding() {

      ParameterSpec.Builder parameterBindingsSpec;
      parameterBindingsSpec = ParameterSpec.builder(ParameterizedTypeName.get(ClassName.get(Map.class),
                                                                              TypeName.get(String.class),
                                                                              TypeName.get(Object.class)),
                                                    CUSTOM_PARAMETER_BINDINGS_NAME);

      ParameterSpec parameterSpec = parameterBindingsSpec.build();

      this.customParameterBinding = new ImmutablePair<>(parameterSpec,
                                                        CodeBlock.builder().add(
                                                                                PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                                                                                "Map that contains custom parameters defined in operation customization"));
      this.finalParameters.add(parameterSpec);
      this.javaDoc.add(this.customParameterBinding.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder contentParameter(Pair<List<ParameterSpec>, CodeBlock.Builder> parameters) {

      if (parameters != null) {
        this.contentParameter = new ImmutablePair<>(parameters.getLeft(),
                                                    CodeBlock.builder().add(parameters.getRight().build()));
        this.finalParameters.addAll(parameters.getLeft());
        this.javaDoc.add(this.contentParameter.getRight().build());
      }

      return this;
    }

    public AbstractSdkOperation.ParametersBuilder requestParameter() {
      ParameterSpec.Builder parameterSpecBuilder = ParameterSpec.builder(RequestParameters.class, "parameters");

      if (withAnnotations) {
        AnnotationSpec parameterGroupAnnotation =
            AnnotationSpec
                .builder(ParameterGroup.class)
                .addMember(NAME_MEMBER, REQUEST_PARAMETERS_GROUP_NAME)
                .build();

        parameterSpecBuilder.addAnnotation(parameterGroupAnnotation);
      }
      ParameterSpec requestParameter = parameterSpecBuilder.build();

      this.requestParameter = new ImmutablePair<>(requestParameter,
                                                  CodeBlock.builder().add(
                                                                          PARAM_DOC_NAME_DESCRIPTION, requestParameter.name,
                                                                          "the {@link "
                                                                              + ((ClassName) requestParameter.type).simpleName()
                                                                              + "}"));
      this.finalParameters.add(requestParameter);
      this.javaDoc.add(this.requestParameter.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder configurationOverridesParameter() {
      ParameterSpec.Builder parameterSpecBuilder = ParameterSpec.builder(ConfigurationOverrides.class, "overrides");

      if (withAnnotations) {
        AnnotationSpec parameterGroupAnnotation =
            AnnotationSpec
                .builder(ParameterGroup.class)
                .addMember(NAME_MEMBER, CONNECTOR_OVERRIDES)
                .build();

        parameterSpecBuilder.addAnnotation(parameterGroupAnnotation);
      }
      ParameterSpec configurationOverride = parameterSpecBuilder.build();

      this.configurationOverridesParameter = new ImmutablePair<>(configurationOverride,
                                                                 CodeBlock.builder().add(PARAM_DOC_NAME_DESCRIPTION,
                                                                                         configurationOverride.name,
                                                                                         "the {@link ConfigurationOverrides}"));
      this.finalParameters.add(configurationOverride);
      this.javaDoc.add(this.configurationOverridesParameter.getRight().build());
      return this;
    }

    public AbstractSdkOperation.ParametersBuilder callbackParameter(MessageOutputType messageOutputType) {
      ParameterSpec completionCallbackParameter = ParameterSpec
          .builder(ParameterizedTypeName.get(ClassName.get(CompletionCallback.class),
                                             messageOutputType.getOutputType(),
                                             messageOutputType.getAttributeOutputType()),
                   "callback")
          .build();

      this.callbackParameter = new ImmutablePair<>(completionCallbackParameter,
                                                   CodeBlock.builder().add(PARAM_DOC_NAME_DESCRIPTION,
                                                                           completionCallbackParameter.name,
                                                                           "the operation's {@link CompletionCallback}"));
      this.finalParameters.add(completionCallbackParameter);
      this.javaDoc.add(this.callbackParameter.getRight().build());
      return this;
    }

    public Pair<List<ParameterSpec>, CodeBlock.Builder> parametersSpecList() {
      return new ImmutablePair<>(finalParameters, javaDoc);
    }
  }
}
