/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.metadata.TypeResolver;
import org.mule.runtime.extension.api.annotation.param.Content;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata.SdkInputMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.FileGenerationUtil;

import java.io.InputStream;
import java.nio.file.Path;
import java.util.List;
import java.util.stream.Collectors;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;

public class SdkAuxiliarParameter extends SdkParameter {

  private SdkInputMetadataResolver inputMetadataResolver = null;
  private final List<SdkField> sdkFields;

  public SdkAuxiliarParameter(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                              ConnectorOperation operation, String parentJavaName, Parameter parameter,
                              JavaTemplateEntity parentElement, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, parentJavaName, parameter, parentElement, runConfiguration);

    final TypeDefinition typeDefinition = parameter.getTypeDefinition();
    if (typeDefinition.getTypeSchema() != null) {
      this.inputMetadataResolver =
          new SdkInputMetadataResolver(outputDir, connectorModel, sdkConnector, operation.getInternalName(), typeDefinition,
                                       FileGenerationUtil.SchemaNameType.INPUT, parameter.getInternalName(), runConfiguration);
    }

    sdkFields = parameter.getFields()
        .stream()
        .map(field -> new SdkField(outputDir, connectorModel, sdkConnector, field, parentElement, runConfiguration))
        .collect(Collectors.toList());
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    super.applyTemplates();
    if (inputMetadataResolver != null) {
      inputMetadataResolver.applyTemplates();
    }
    for (SdkField sdkField : sdkFields) {
      sdkField.applyTemplates();
    }
  }

  @Override
  protected void configureParameterSpecBuilder(ParameterSpec.Builder paramSpecBuilder) {
    super.configureParameterSpecBuilder(paramSpecBuilder);

    if (inputMetadataResolver != null && inputMetadataResolver.getRequiresMetadataResolver()) {
      paramSpecBuilder
          .addAnnotation(generateMetadataResolverAnnotation())
          .addAnnotation(generateContentAnnotation());

      sdkFields
          .forEach(sdkField -> paramSpecBuilder.addAnnotation(getValueProviderAnnotation(sdkField, parameter.getInternalName())));
    }
  }

  private AnnotationSpec generateMetadataResolverAnnotation() {
    return AnnotationSpec
        .builder(TypeResolver.class)
        .addMember("value", "$T.class", ClassName.get(inputMetadataResolver.getPackage(), inputMetadataResolver.getClassName()))
        .build();
  }

  private AnnotationSpec generateContentAnnotation() {
    return AnnotationSpec
        .builder(Content.class)
        .build();
  }

  @Override
  public TypeName getTypeName() {
    return inputMetadataResolver != null ? ParameterizedTypeName.get(TypedValue.class, InputStream.class) : super.getTypeName();
  }
}
