/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.util;

import static com.google.common.base.Charsets.UTF_8;
import static java.lang.String.format;
import static org.apache.commons.io.FileUtils.writeStringToFile;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.JsonTypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.XmlTypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;

import org.json.JSONObject;

public class FileGenerationUtil {

  public static void writeSchema(TypeSchema source, Path outputDir, String schemaName) {
    try {

      File file = outputDir.resolve(schemaName).toFile();

      if (file.exists()) {
        return;
      }

      String schemaString = source.getRawSchema();

      if (source instanceof JsonTypeSchema) {
        JSONObject jObject = new JSONObject(schemaString);
        schemaString = jObject.toString(2);
      }
      writeStringToFile(file, schemaString, UTF_8);

    } catch (IOException e) {
      throw new IllegalArgumentException(format("Path %s is invalid.", outputDir), e);
    }
  }

  public enum SchemaNameType {
    INPUT, OUTPUT, PART
  }

  public static String generateSchemaName(TypeSchema source, String internalName, SchemaNameType nameType,
                                          String partName) {
    String suffix = getSchemaSuffix(source, nameType, partName);
    return internalName + suffix;
  }

  private static String getSchemaSuffix(TypeSchema source, SchemaNameType nameType, String partName) {
    String schemaSuffix = EMPTY;

    switch (nameType) {
      case INPUT:
        schemaSuffix = "-input-schema.";
        break;
      case OUTPUT:
        schemaSuffix = "-output-schema.";
        break;
      case PART:
        schemaSuffix = "-part-input-schema.";
        break;
    }

    if (isNotBlank(partName)) {
      schemaSuffix = "-" + XmlUtils.getXmlName(partName) + schemaSuffix;
    }

    return schemaSuffix + getExtension(source);
  }

  private static String getExtension(TypeSchema source) {
    if (source instanceof XmlTypeSchema) {
      return "xsd";
    }

    if (source instanceof JsonTypeSchema) {
      return "json";
    }

    throw new IllegalArgumentException("Type Schema class doesn't support typeSchema generation.");
  }

  public static boolean supportsSchemaGeneration(TypeSchema source) {
    return source instanceof XmlTypeSchema
        || source instanceof JsonTypeSchema;
  }
}
