/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getParameterJavaName;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.getXmlName;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;

import com.mulesoft.connectivity.rest.commons.internal.model.builder.common.EvaluationContextBuilderFactory;
import com.mulesoft.connectivity.rest.commons.internal.model.common.EvaluationContext;
import com.mulesoft.connectivity.rest.commons.internal.model.dataexpressions.BindingField;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;

import java.nio.file.Path;
import java.util.List;

import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

public abstract class AbstractSdkResolverProvider extends JavaTemplateEntity {

  public AbstractSdkResolverProvider(Path outputDir,
                                     ConnectorModel connectorModel,
                                     RestSdkRunConfiguration runConfiguration) {

    super(outputDir, connectorModel, runConfiguration);
  }

  public abstract String getPackage();

  public abstract String getJavaClassName();

  @Override
  public void applyTemplates() throws TemplatingException {
    generateClass();
  }

  protected void generateClass() throws TemplatingException {
    TypeSpec.Builder classBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(getSuperClass());

    buildClass(classBuilder);

    JavaFile.Builder javaFileBuilder = getJavaFileBuilderForClass(classBuilder.build(), getPackage());
    writeJavaFile(javaFileBuilder.build());
  }

  /**
   * Returns the super class the generated value provider will extend. This is useful when multiple value providers extend a
   * commons parent one where the logic is implemented. i.e. Reference.
   */
  protected abstract TypeName getSuperClass();

  /**
   * Allows extending the default value provider class building
   */
  protected abstract void buildClass(TypeSpec.Builder classBuilder);

  /**
   * Indicates if a {@link org.mule.runtime.extension.api.annotation.param.Parameter} from the parent operation of the value
   * provider is necessary for this value provider to resolve the available values. If this method says is necessary, a @Parameter
   * will be generated in the value provider class. When that @Parameter is generated in the value provider, it is mandatory for
   * it to have an assigned value in the DSL for the resolver to be executed.
   *
   * @param sdkParameter The parameter that is being checked
   */
  protected abstract boolean isBoundParameter(SdkParameter sdkParameter);

  /**
   * Returns all the parameters - regardless of its type - for this value resolver. These will later be used generate bindings
   * and @Parameters if needed.
   */
  protected abstract List<SdkParameter> getAllParameters();

  /**
   * Returns all the fields of the body necessaries for this value provider. These will later be used generate bindings
   * and @Parameters if needed.
   */
  protected abstract List<BindingField> getBindingFields();

  /**
   * Returns the kind os evaluation context that need to be build (i.e. Trigger/Operation)
   */
  protected abstract String getEvaluationContextKind();

  /**
   * Adds all the bound parameters - as {@link #isBoundParameter} indicates - to the classBuilder as @Parameters.
   */
  protected void addParameterFieldsIfNeeded(TypeSpec.Builder classBuilder) {
    for (SdkParameter sdkParameter : getAllParameters()) {
      if (isBoundParameter(sdkParameter)) {
        classBuilder.addField(getParameterFieldSpec(sdkParameter));
      }
    }

    for (BindingField field : getBindingFields()) {
      classBuilder.addField(getParameterFieldSpec(field));
    }

  }

  /**
   * Generates a field annotated with @Parameter corresponding to the provided SdkParameter.
   */
  private FieldSpec getParameterFieldSpec(SdkParameter sdkParameter) {
    FieldSpec.Builder fieldBuilder =
        FieldSpec.builder(sdkParameter.getTypeName(),
                          sdkParameter.getJavaName(),
                          PRIVATE)
            .addAnnotation(org.mule.runtime.extension.api.annotation.param.Parameter.class);

    return fieldBuilder.build();
  }

  private FieldSpec getParameterFieldSpec(BindingField field) {
    FieldSpec.Builder fieldBuilder =
        FieldSpec.builder(TypeName.get(String.class),
                          getParameterJavaName(getXmlName(field.getName()),
                                               false),
                          PRIVATE)
            .addAnnotation(org.mule.runtime.extension.api.annotation.param.Parameter.class);

    return fieldBuilder.build();
  }

  protected void addConfigureEvaluationContextMethod(TypeSpec.Builder classBuilder) {
    final MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder("buildEvaluationContext")
            .addModifiers(PROTECTED)
            .returns(EvaluationContext.class)
            .addParameter(TypeName.get(EvaluationContextBuilderFactory.class), "builderFactory")
            .addAnnotation(Override.class);

    addParameterBindings(methodBuilder, getAllParameters(), getBindingFields());

    classBuilder.addMethod(methodBuilder.build());
  }

  private void addParameterBindings(MethodSpec.Builder methodBuilder, List<SdkParameter> parameters, List<BindingField> fields) {
    int count = 0;

    CodeBlock.Builder contextBindingBuilder = CodeBlock.builder();

    contextBindingBuilder.add("return builderFactory.");

    for (SdkParameter sdkParameter : parameters) {
      if (isBoundParameter(sdkParameter)) {
        if (count == 0) {
          count++;
          contextBindingBuilder.add("$LContextBuilder()", getEvaluationContextKind());
        }

        contextBindingBuilder.add(".$1L($2S, $3L)",
                                  sdkParameter.getParameterType().getAccessorName(),
                                  sdkParameter.getExternalName(),
                                  sdkParameter.getJavaName());
      }
    }

    for (BindingField field : fields) {
      if (count == 0) {
        count++;
        contextBindingBuilder.add("$LContextBuilder()", getEvaluationContextKind());
      }

      contextBindingBuilder.add(".$1L($2S, $3L)",
                                ParameterType.BODY.getAccessorName(),
                                field.getValue(),
                                getParameterJavaName(getXmlName(field.getName()),
                                                     false));
    }

    if (count == 0) {
      contextBindingBuilder.add("emptyContextBuilder()", getEvaluationContextKind());
    }

    contextBindingBuilder.add(".build()");

    methodBuilder.addStatement(contextBindingBuilder.build());
  }
}
