/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isBlank;
import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkContent.SdkContentKind.INPUT_METADATA;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.ContentNameGenerator.getContentName;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.metadata.TypeResolver;
import org.mule.runtime.extension.api.annotation.param.Content;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata.SdkInputMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.FileGenerationUtil;

import java.io.InputStream;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import org.apache.commons.lang3.StringUtils;

public class SdkContent extends JavaTemplateEntity {

  public static final String BODY_DISPLAY_NAME = "Body";
  private final SdkInputMetadataResolver inputMetadataResolver;
  private final String contentParameterJavaName;
  private final Optional<Boolean> primary;
  private final TypeDefinition typeDefinition;
  private final String displayName;
  private final List<SdkField> sdkFields;
  private final SdkConnector sdkConnector;
  private final JavaTemplateEntity parentElement;

  public enum SdkContentKind {
    INPUT_METADATA, PART
  }

  public SdkContent(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, ConnectorOperation operation,
                    JavaTemplateEntity parentElement, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    this(outputDir, connectorModel, sdkConnector, operation, operation.getInputMetadata(), INPUT_METADATA, EMPTY, EMPTY,
         Optional.ofNullable(true),
         parentElement, runConfiguration);
  }

  public SdkContent(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, ConnectorOperation operation,
                    TypeDefinition typeDefinition, SdkContentKind contentType, String partName, String partIdentifier,
                    Optional<Boolean> primary,
                    JavaTemplateEntity parentElement, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);
    this.typeDefinition = typeDefinition;

    FileGenerationUtil.SchemaNameType schemaNameType =
        contentType.equals(INPUT_METADATA) ? FileGenerationUtil.SchemaNameType.INPUT : FileGenerationUtil.SchemaNameType.PART;

    this.inputMetadataResolver =
        new SdkInputMetadataResolver(outputDir, connectorModel, sdkConnector, operation.getInternalName(), typeDefinition,
                                     schemaNameType,
                                     partName, runConfiguration);

    this.contentParameterJavaName = getContentName(operation, inputMetadataResolver,
                                                   contentType,
                                                   partIdentifier);

    this.primary = primary;
    this.displayName = buildDisplayName(schemaNameType, partName);
    this.sdkConnector = sdkConnector;
    this.parentElement = parentElement;

    if (operation.getBody() != null) {
      sdkFields = operation.getBody().getFields()
          .stream()
          .map(field -> new SdkField(outputDir, connectorModel, sdkConnector, field, parentElement, runConfiguration))
          .collect(Collectors.toList());
    } else {
      sdkFields = new ArrayList<>();
    }
  }

  private String buildDisplayName(FileGenerationUtil.SchemaNameType schemaNameType, String partName) {
    String result;
    if (FileGenerationUtil.SchemaNameType.INPUT.equals(schemaNameType)) {
      // body simple/single
      result = BODY_DISPLAY_NAME;
    } else {
      // body multipart, if no display name is provided, will default to partName identifier
      String partDisplayName =
          isNotBlank(typeDefinition.getDisplayName()) ? typeDefinition.getDisplayName()
              : StringUtils.capitalize(StringUtils.join(StringUtils.splitByCharacterTypeCamelCase(partName), StringUtils.SPACE));
      result = String.format("%s - %s", partDisplayName, BODY_DISPLAY_NAME);
    }
    return result;
  }

  private TypeName getTypeName() {
    return ParameterizedTypeName.get(TypedValue.class, InputStream.class);
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    inputMetadataResolver.applyTemplates();
    for (SdkField sdkField : sdkFields) {
      sdkField.applyTemplates();
    }
  }

  public ParameterSpec generateContentParameter() {
    ParameterSpec.Builder paramSpecBuilder = ParameterSpec
        .builder(getTypeName(), contentParameterJavaName)
        .addAnnotation(generateContentAnnotation());

    AnnotationSpec displayNameAnnotation = generateDisplayNameAnnotation();
    if (displayNameAnnotation != null) {
      paramSpecBuilder.addAnnotation(displayNameAnnotation);
    }

    AnnotationSpec summaryAnnotation = generateSummaryAnnotation();
    if (summaryAnnotation != null) {
      paramSpecBuilder.addAnnotation(summaryAnnotation);
    }

    if (inputMetadataResolver != null && inputMetadataResolver.getRequiresMetadataResolver()) {
      paramSpecBuilder.addAnnotation(generateMetadataResolverAnnotation());
    }

    sdkFields.forEach(sdkField -> paramSpecBuilder.addAnnotation(getValueProviderAnnotation(sdkField, contentParameterJavaName)));

    return paramSpecBuilder.build();
  }

  private AnnotationSpec generateDisplayNameAnnotation() {
    if (isBlank(displayName)) {
      return null;
    }

    return AnnotationSpec
        .builder(DisplayName.class)
        .addMember("value", "$S", displayName)
        .build();
  }

  private AnnotationSpec generateSummaryAnnotation() {
    String summary = typeDefinition.getDescription();

    if (isBlank(summary)) {
      return null;
    }

    return AnnotationSpec
        .builder(Summary.class)
        .addMember("value", "$S", summary)
        .build();
  }

  private AnnotationSpec generateMetadataResolverAnnotation() {
    return AnnotationSpec
        .builder(TypeResolver.class)
        .addMember("value", "$T.class", ClassName.get(inputMetadataResolver.getPackage(), inputMetadataResolver.getClassName()))
        .build();
  }

  private AnnotationSpec generateContentAnnotation() {
    AnnotationSpec.Builder builder = AnnotationSpec
        .builder(Content.class);
    primary.ifPresent(aBoolean -> {
      builder.addMember("primary", "$L", aBoolean);
    });
    return builder.build();
  }

  public String getContentParameterJavaName() {
    return contentParameterJavaName;
  }

}
