/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.operation;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.AUXILIAR;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.BODY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition.simplePrimitiveType;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.abbreviateText;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaLowerCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.SdkResolverUtil.getActingParameterJavaName;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.sampledata.SdkSampleDataFactory.getSdkSampleDataResolver;
import static com.mulesoft.connectivity.rest.sdk.templating.sdk.util.SdkBodyLevelUtils.transformBodyPrefix;
import static java.util.Arrays.stream;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static javax.lang.model.element.Modifier.ABSTRACT;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.PROTECTED;
import static javax.lang.model.element.Modifier.PUBLIC;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA_TYPE;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import org.mule.runtime.extension.api.annotation.Alias;
import org.mule.runtime.extension.api.annotation.Ignore;
import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.metadata.OutputResolver;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.HttpConstants;
import org.mule.sdk.api.annotation.binding.Binding;
import org.mule.sdk.api.annotation.data.sample.SampleData;

import com.mulesoft.connectivity.rest.commons.api.binding.HttpRequestBinding;
import com.mulesoft.connectivity.rest.commons.api.binding.HttpResponseBinding;
import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.datasense.metadata.output.SdkHttpResponseAttributesMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.error.RequestErrorTypeProvider;
import com.mulesoft.connectivity.rest.commons.api.operation.BaseRestOperation;
import com.mulesoft.connectivity.rest.commons.api.operation.ConfigurationOverrides;
import com.mulesoft.connectivity.rest.commons.api.operation.EntityRequestParameters;
import com.mulesoft.connectivity.rest.commons.api.operation.NonEntityRequestParameters;
import com.mulesoft.connectivity.rest.commons.internal.RestConstants;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder.QueryParamFormat;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.body.Body;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.body.Field;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.dataexpression.httprequest.HttpRequestDataExpression;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.Argument;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.PaginationType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.AuxiliarParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.resolver.ResolverReference;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.sampledata.SampleDataDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata.SdkOutputMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.metadata.SdkPagingMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.munit.SdkMtfOperationTest;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkAuxiliarParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkContent;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkField;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkPaginationParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.resolver.AbstractSdkResolverProvider;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.util.MuleAnnotationsUtils;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import java.io.InputStream;
import java.lang.reflect.TypeVariable;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.lang.model.element.Modifier;

import org.apache.commons.lang3.StringUtils;

public abstract class AbstractSdkOperation extends JavaTemplateEntity {

  public static final String PARAM_DOC_NAME_DESCRIPTION = "@param $L $L\n";

  public static final String CONTENT_TYPE_HEADER_NAME = "content-type";
  public static final String ACCEPT_HEADER_NAME = "accept";

  public static final String ADD_URI_PARAM_METHOD_NAME = "addUriParam";
  public static final String ADD_QUERY_PARAM_METHOD_NAME = "addQueryParam";
  public static final String ADD_HEADER_METHOD_NAME = "addHeader";

  public static final String ADD_MULTIPLE_URI_PARAM_METHOD_NAME = "addUriParams";
  public static final String ADD_MULTIPLE_QUERY_PARAM_METHOD_NAME = "addQueryParams";
  public static final String ADD_MULTIPLE_HEADER_METHOD_NAME = "addHeaders";

  private static final String REQUEST_PARAMETERS_GROUP_NAME = "REQUEST_PARAMETERS_GROUP_NAME";
  private static final String CONNECTOR_OVERRIDES = "CONNECTOR_OVERRIDES";

  private static final String GET_REQUEST_BODY_DATA_TYPE_METHOD = "getRequestBodyMediaType";
  private static final String GET_RESPONSE_BODY_DATA_TYPE_METHOD = "getResponseBodyMediaType";

  private static final String QUERY_PARAM_FORMAT_FIELD = "QUERY_PARAM_FORMAT";
  public static final String OPERATION_PATH_FIELD = "OPERATION_PATH";

  public static final String INTERCEPTOR_METHOD_NAME = "doBeforeRequest";

  protected final ConnectorOperation operation;
  private final SdkConnector sdkConnector;
  protected final List<SdkParameter> allUriParameters;
  protected final List<SdkParameter> allQueryParameters;
  protected final List<SdkParameter> allHeaders;
  protected final List<SdkParameter> auxParameters;
  protected final List<SdkField> allBodyFields;
  protected final SdkContent content;
  private final Optional<String> muleOutputResolver;
  protected final SdkOutputMetadataResolver outputMetadataResolver;

  protected final AbstractSdkResolverProvider sampleDataProvider;

  private final SdkMtfOperationTest sdkMtfOperationTest;

  public abstract TypeName generateMethodReturn();

  public abstract CodeBlock generateOperationMethodBody(List<ParameterSpec> parameters) throws TemplatingException;

  public AbstractSdkOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                              ConnectorOperation operation, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);

    this.operation = operation;
    this.sdkConnector = sdkConnector;

    this.allUriParameters = buildSdkParameters(outputDir, connectorModel, sdkConnector, operation.getUriParameters(), false);
    this.allQueryParameters = buildSdkParameters(outputDir, connectorModel, sdkConnector, operation.getQueryParameters(), true);
    this.allHeaders = buildSdkParameters(outputDir, connectorModel, sdkConnector, operation.getHeaders(), false);
    this.auxParameters = new ArrayList<>();
    final List<AuxiliarParameter> parameters = operation.getParameters().orElse(null);
    if (parameters != null) {
      this.auxParameters
          .addAll(buildAuxSdkParameters(operation, parameters));
    }
    this.allBodyFields = buildSdkFields(outputDir, connectorModel, sdkConnector, operation.getBody());

    this.content = buildContent(outputDir, connectorModel, sdkConnector, operation);

    this.muleOutputResolver = Optional.ofNullable(StringUtils.trim(operation.getMuleOutputResolver()));
    this.outputMetadataResolver = (muleOutputResolver.isPresent()) ? null : buildOutputMetadataResolver(sdkConnector);

    this.sdkMtfOperationTest = new SdkMtfOperationTest(operation, outputDir);
    this.sampleDataProvider = buildSdkSampleDataProvider(operation);
  }

  private AbstractSdkResolverProvider buildSdkSampleDataProvider(ConnectorOperation operation) {
    return getSdkSampleDataResolver(operation, connectorModel, sdkConnector, getJavaClassName(), getAllParameters(), outputDir,
                                    runConfiguration);
  }

  private List<SdkField> buildSdkFields(Path outputDir,
                                        ConnectorModel connectorModel,
                                        SdkConnector sdkConnector,
                                        Body body) {
    final List<SdkField> list = new ArrayList<>();
    if (body == null) {
      return list;
    }

    for (Field field : body.getFields()) {
      list.add(new SdkField(outputDir, connectorModel, sdkConnector, field, this, runConfiguration));
    }

    return list;
  }

  private List<SdkParameter> buildSdkParameters(Path outputDir,
                                                ConnectorModel connectorModel,
                                                SdkConnector sdkConnector,
                                                List<Parameter> connectorParameters,
                                                boolean checkPaginationParameters) {
    final List<SdkParameter> list = new ArrayList<>();

    for (Parameter parameter : connectorParameters) {
      if (checkPaginationParameters && isQueryParamDefinedInPagination(parameter.getExternalName())) {
        Pagination pagination = getPagination();

        list.add(new SdkPaginationParameter(outputDir, connectorModel, sdkConnector, getJavaClassName(),
                                            parameter, this, runConfiguration, pagination));
      } else {
        list.add(new SdkParameter(outputDir, connectorModel, sdkConnector, getJavaClassName(), parameter, this,
                                  runConfiguration));
      }
    }

    return list;
  }

  private List<SdkParameter> buildAuxSdkParameters(ConnectorOperation operation,
                                                   List<AuxiliarParameter> connectorAuxParameters)
      throws TemplatingException {
    final List<SdkParameter> list = new ArrayList<>();
    if (connectorAuxParameters != null) {
      for (AuxiliarParameter parameter : connectorAuxParameters) {
        list.add(buildSdkParameter(parameter, operation));
      }
    }
    return list;
  }

  private SdkParameter buildSdkParameter(AuxiliarParameter auxiliarParameter, ConnectorOperation operation)
      throws TemplatingException {
    TypeDefinition typeDefinition = auxiliarParameter.getTypeDefinition();
    if (typeDefinition == null) {
      typeDefinition = getTypeDefinition(auxiliarParameter.getType());
    }
    com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter parameter =
        new com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter(
                                                                                           auxiliarParameter.getDisplayName(),
                                                                                           auxiliarParameter.getExternalName(),
                                                                                           auxiliarParameter.getExternalName(),
                                                                                           AUXILIAR,
                                                                                           typeDefinition,
                                                                                           auxiliarParameter.getDescription(),
                                                                                           auxiliarParameter.isRequired() == null
                                                                                               || auxiliarParameter.isRequired(),
                                                                                           null,
                                                                                           false,
                                                                                           auxiliarParameter.getValueProvider(),
                                                                                           auxiliarParameter
                                                                                               .getMuleMetadataKeyId(),
                                                                                           auxiliarParameter
                                                                                               .getMuleTypeResolver(),
                                                                                           auxiliarParameter.getFields());

    return new SdkAuxiliarParameter(outputDir, connectorModel, sdkConnector, operation, getJavaClassName(), parameter, this,
                                    runConfiguration);
  }

  private TypeDefinition getTypeDefinition(ParameterDataType parameterDataType) {
    PrimitiveTypeDefinition.PrimitiveType primitiveType;

    switch (parameterDataType) {
      case NUMBER:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.NUMBER;
        break;
      case INTEGER:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.INTEGER;
        break;
      case LONG:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.LONG;
        break;
      case STRING:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.STRING;
        break;
      case BOOLEAN:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.BOOLEAN;
        break;
      case LOCAL_DATE_TIME:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.LOCAL_DATE_TIME;
        break;
      case ZONED_DATE_TIME:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.ZONED_DATE_TIME;
        break;
      case BINARY:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.BINARY;
        break;
      default:
        throw new IllegalArgumentException("Parameter type not suppoerted. This is a bug.");
    }

    return simplePrimitiveType(primitiveType);
  }

  public ConnectorOperation getOperation() {
    return operation;
  }

  private SdkOutputMetadataResolver buildOutputMetadataResolver(SdkConnector sdkConnector) throws TemplatingException {
    if (operation.getOutputMetadata() == null) {
      return null;
    }

    if (operation.hasPagination()) {
      return new SdkPagingMetadataResolver(outputDir,
                                           connectorModel,
                                           sdkConnector,
                                           operation.getInternalName(),
                                           operation.getOutputMetadata(),
                                           runConfiguration);
    } else {
      return new SdkOutputMetadataResolver(outputDir,
                                           connectorModel,
                                           sdkConnector,
                                           operation.getInternalName(),
                                           operation.getOutputMetadata(),
                                           EMPTY,
                                           runConfiguration);
    }
  }

  protected SdkContent buildContent(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                    ConnectorOperation operation)
      throws TemplatingException {

    if (operation.isRequestBodyBound()) {
      return null;
    }
    return operation.getInputMetadata() != null
        ? new SdkContent(outputDir, connectorModel, sdkConnector, operation, this, runConfiguration)
        : null;
  }

  public String getJavaClassName() {
    return getJavaUpperCamelNameFromXml(operation.getInternalName()) + "Operation";
  }

  public String getJavaBaseClassName() {
    return getJavaUpperCamelNameFromXml(operation.getInternalName()) + "OperationBase";
  }

  public String getJavaInterceptorClassName() {
    return getJavaUpperCamelNameFromXml(operation.getInternalName()) + "OperationInterceptor";
  }

  private String getJavaMethodName() {
    return getJavaLowerCamelNameFromXml(operation.getInternalName());
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + ".internal.operation";
  }

  public String getInterceptorPackage() {
    return connectorModel.getBasePackage() + ".internal.operation.interceptor";
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (content != null) {
      content.applyTemplates();
    }

    if (outputMetadataResolver != null) {
      outputMetadataResolver.applyTemplates();
    }

    for (SdkParameter sdkParameter : allUriParameters) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : allQueryParameters) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : allHeaders) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : auxParameters) {
      sdkParameter.applyTemplates();
    }

    if (!runConfiguration.regenerateMode()) {
      sdkMtfOperationTest.applyTemplates();
    }
    if (sampleDataProvider != null) {
      sampleDataProvider.applyTemplates();
    }

    if (operation.isAdapter()) {
      generateOperationLowerClass();
      generateOperationMiddleClass();
      generateOperationHigherClass();
    } else {
      generateOperationClass();
    }
  }

  private MethodSpec generateGetRequestBodyDataTypeMethod() {
    return generateGetBodyDataTypeMethod(GET_REQUEST_BODY_DATA_TYPE_METHOD, getRequestBodyMediaType());
  }

  private String getRequestBodyMediaType() {
    return getBodyMediaType(getOperation().getInputMetadata());
  }

  private MethodSpec generateGetResponseBodyDataTypeMethod() {
    return generateGetBodyDataTypeMethod(GET_RESPONSE_BODY_DATA_TYPE_METHOD, getResponseBodyMediaType());
  }

  private String getResponseBodyMediaType() {
    return getBodyMediaType(getOperation().getOutputMetadata());
  }

  private String getBodyMediaType(TypeDefinition typeDefinition) {
    String result;
    if (typeDefinition != null && typeDefinition.getMediaType() != null) {
      result = typeDefinition.getMediaType().toString();
    } else {
      // Default value
      result = MediaType.APPLICATION_JSON.toString();
    }
    return result;
  }

  private MethodSpec generateGetBodyDataTypeMethod(String name, String bodyMediaType) {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(name)
            .addModifiers(PROTECTED)
            .returns(String.class)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("return \"$L\"", bodyMediaType);

    return methodBuilder.build();
  }

  protected void generateOperationClass() throws TemplatingException {
    TypeSpec.Builder operationClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(getSuperclass())
            .addMethod(generateOperationMethod());

    defineConstructors(operationClassBuilder);

    operationClassBuilder.addField(generateOperationPathField());
    operationClassBuilder.addField(generateQueryParamFormatField());

    configureClassBuilder(operationClassBuilder);
    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getPackage(), operationClassBuilder.build())
        .skipJavaLangImports(true)
        .addStaticImport(RestConstants.class, REQUEST_PARAMETERS_GROUP_NAME, CONNECTOR_OVERRIDES);

    configureJavaFileBuilder(javaFileBuilder);

    writeJavaFile(javaFileBuilder.build());
  }

  protected void generateOperationLowerClass() throws TemplatingException {
    TypeSpec.Builder operationClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(PUBLIC)
            .superclass(ClassName.get(getInterceptorPackage(), getJavaInterceptorClassName()))
            .addJavadoc("Lower part of the Operation. It has the operation declaration with its annotations.")
            .addMethod(generateOperationMethodLower());

    defineConstructors(operationClassBuilder);

    configureClassBuilder(operationClassBuilder);
    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getPackage(), operationClassBuilder.build())
        .skipJavaLangImports(true)
        .addStaticImport(RestConstants.class, REQUEST_PARAMETERS_GROUP_NAME, CONNECTOR_OVERRIDES);

    configureJavaFileBuilder(javaFileBuilder);

    writeJavaFile(javaFileBuilder.build());
  }

  protected void generateOperationMiddleClass() throws TemplatingException {
    TypeSpec.Builder operationClassBuilder =
        TypeSpec
            .classBuilder(getJavaInterceptorClassName())
            .addModifiers(PUBLIC)
            .superclass(ClassName.get(getPackage(), getJavaBaseClassName()))
            .addJavadoc("Middle part of the Operation. Can be used by the user to add custom code into the operation.")
            .addMethod(generateOperationMethodInterceptor());

    defineConstructors(operationClassBuilder);

    configureClassBuilder(operationClassBuilder);
    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getInterceptorPackage(), operationClassBuilder.build())
        .skipJavaLangImports(true);

    configureJavaFileBuilder(javaFileBuilder);

    writeJavaFile(javaFileBuilder.build(), true);
  }

  protected void generateOperationHigherClass() throws TemplatingException {
    MethodSpec operationMethod = generateOperationMethodHigher();

    TypeSpec.Builder operationClassBuilder =
        TypeSpec
            .classBuilder(getJavaBaseClassName())
            .addModifiers(PUBLIC)
            .addModifiers(ABSTRACT)
            .superclass(getSuperclass())
            .addJavadoc("Higher part of the Operation. It has the implementation of the operation.")
            .addMethod(operationMethod);

    defineConstructors(operationClassBuilder);

    operationClassBuilder.addField(generateOperationPathField());
    operationClassBuilder.addField(generateQueryParamFormatField());

    operationClassBuilder.addMethod(generateDoBeforeRequestMethod(operationMethod.parameters));
    operationClassBuilder.addMethod(generateGetRequestBodyDataTypeMethod());
    operationClassBuilder.addMethod(generateGetResponseBodyDataTypeMethod());
    operationClassBuilder.addMethod(generateRequestBindingsMethod());
    operationClassBuilder.addMethod(generateResponseBindingsMethod());

    configureClassBuilder(operationClassBuilder);
    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getPackage(), operationClassBuilder.build())
        .skipJavaLangImports(true);

    configureJavaFileBuilder(javaFileBuilder);

    writeJavaFile(javaFileBuilder.build());
  }

  private void defineConstructors(TypeSpec.Builder operationClassBuilder) {
    stream(getSuperclass().getDeclaredConstructors()).forEach(constructor -> {
      MethodSpec.Builder constructorBuilder = MethodSpec.constructorBuilder().addModifiers(Modifier.PUBLIC);
      StringBuilder methodStatement = new StringBuilder("super(");
      for (int i = 0; i < constructor.getParameterTypes().length; i++) {
        Class parameterType = constructor.getParameterTypes()[i];
        String name = "arg" + i;
        constructorBuilder.addParameter(parameterType, name);
        if (i > 0) {
          methodStatement.append(", ");
        }
        methodStatement.append(name);
      }
      methodStatement.append(")");
      constructorBuilder.addStatement(methodStatement.toString());
      operationClassBuilder.addMethod(constructorBuilder.build());
    });
  }

  protected void configureJavaFileBuilder(JavaFile.Builder javaFileBuilder) {}

  protected void configureClassBuilder(TypeSpec.Builder operationClassBuilder) {}

  protected Class<? extends BaseRestOperation> getSuperclass() {
    return BaseRestOperation.class;
  }

  private FieldSpec generateQueryParamFormatField() {
    return FieldSpec.builder(QueryParamFormat.class,
                             QUERY_PARAM_FORMAT_FIELD,
                             operation.isAdapter() ? PROTECTED : PRIVATE,
                             Modifier.FINAL,
                             Modifier.STATIC)
        .initializer("$T.$L", QueryParamFormat.class, operation.getQueryParamArrayFormat().name())
        .build();
  }

  private FieldSpec generateOperationPathField() {
    return FieldSpec.builder(String.class,
                             OPERATION_PATH_FIELD,
                             operation.isAdapter() ? PROTECTED : PRIVATE,
                             Modifier.FINAL,
                             Modifier.STATIC)
        .initializer("$S", operation.getPath())
        .build();
  }

  private AnnotationSpec generateDescriptionAnnotation() {
    return AnnotationSpec
        .builder(Summary.class)
        .addMember(VALUE_MEMBER, "$S", abbreviateText(operation.getDescription()))
        .build();
  }

  public MethodSpec generateOperationMethod() throws TemplatingException {

    CodeBlock.Builder javaDoc = generateJavadoc();

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(getJavaMethodName())
        .addModifiers(PUBLIC);

    addParameters(methodBuilder, javaDoc, true);

    addAnnotations(methodBuilder);

    methodBuilder.addCode(generateOperationMethodBody(methodBuilder.parameters));

    methodBuilder.addJavadoc(javaDoc.build());

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  private CodeBlock.Builder generateJavadoc() {
    return CodeBlock.builder()
        .add("\n$L\n", defaultIfEmpty(operation.getDescription(), operation.getDisplayName()))
        .add("\nThis operation makes an HTTP $L request to the $L endpoint", operation.getHttpMethod().name().toUpperCase(),
             operation.getPath())
        .add("\n");
  }

  private void addAnnotations(MethodSpec.Builder methodBuilder) {
    methodBuilder.addAnnotation(generateThrowsAnnotation());
    methodBuilder.addAnnotation(generateDisplayNameAnnotation());

    if (operation.isSidecar()) {
      methodBuilder.addAnnotation(generateSidecarAnnotation());
    }

    if (isNotBlank(operation.getDescription())) {
      methodBuilder.addAnnotation(generateDescriptionAnnotation());
    }

    if (isNotBlank(operation.getAlias())) {
      methodBuilder.addAnnotation(generateAliasAnnotation());
    }

    if (sampleDataProvider != null) {
      methodBuilder.addAnnotation(generateSampleDataAnnotation());
    }

    if (requiresMediaTypeAnnotation()) {
      if (operation.getOutputMetadata() != null) {
        methodBuilder.addAnnotation(generateMediaTypeAnnotation());
      } else {
        methodBuilder.addAnnotation(generateDefaultMediaTypeAnnotation());
      }
    }

    methodBuilder.addAnnotation(getOutputResolverAnnotation());
  }

  public MethodSpec generateOperationMethodLower() throws TemplatingException {

    CodeBlock.Builder javaDoc = generateJavadoc();

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(getJavaMethodName())
        .addModifiers(PUBLIC);

    addParameters(methodBuilder, javaDoc, true);

    addAnnotations(methodBuilder);

    methodBuilder.addCode(generateOperationLowerMethodBody(methodBuilder.parameters));

    methodBuilder.addJavadoc(javaDoc.build());

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  public MethodSpec generateOperationMethodHigher() throws TemplatingException {

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(getJavaMethodName())
        .addModifiers(PROTECTED);

    addParameters(methodBuilder, CodeBlock.builder(), false);

    methodBuilder.addCode(generateOperationMethodBody(methodBuilder.parameters));

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  public MethodSpec generateOperationMethodInterceptor() throws TemplatingException {

    CodeBlock.Builder javaDoc = CodeBlock.builder()
        .add("Method given to add custom code to be run before the HTTP request")
        .add("\n");

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(INTERCEPTOR_METHOD_NAME)
        .addModifiers(PROTECTED)
        .addAnnotation(Override.class);

    addParameters(methodBuilder, javaDoc, false);

    methodBuilder.addCode(generateOperationInterceptorMethodBody());

    methodBuilder.addJavadoc(javaDoc.build());

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  private void addParameters(MethodSpec.Builder methodBuilder, CodeBlock.Builder javaDoc, boolean withAnnotations) {
    ParameterSpec configParameter = generateConfigParameter(withAnnotations);
    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, configParameter.name, "the configuration to use");
    methodBuilder.addParameter(configParameter);

    if (requiresConnectionParameter()) {
      ParameterSpec connectionParameter = generateConnectionParameter(withAnnotations);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, connectionParameter.name, "the connection to use");
      methodBuilder.addParameter(connectionParameter);
    }

    for (SdkParameter sdkParam : allUriParameters) {
      ParameterSpec parameterSpec = sdkParam.generateParameterParameter(withAnnotations).build();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                  defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
    }

    // Exclude param with same param name that token name marker pagination
    List<SdkParameter> filteredQueryParameters = allQueryParameters;
    if (isEndpointMarkerPagination()) {
      filteredQueryParameters = allQueryParameters.stream()
          .filter(param -> !isQueryParamDefinedInMarkerPagination(param.getExternalName()))
          .collect(toList());
    }
    for (SdkParameter sdkParam : filteredQueryParameters) {
      ParameterSpec parameterSpec = sdkParam.generateParameterParameter(withAnnotations).build();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                  defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
    }

    for (SdkParameter sdkParam : allHeaders) {
      ParameterSpec parameterSpec = sdkParam.generateParameterParameter(withAnnotations).build();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                  defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
    }

    for (SdkParameter sdkParam : auxParameters) {
      ParameterSpec parameterSpec = sdkParam.generateParameterParameter(withAnnotations).build();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                  defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
    }

    addContentParameters(javaDoc, methodBuilder, withAnnotations);

    ParameterSpec parameterSpec = generateRequestParametersParameter(withAnnotations);
    methodBuilder.addParameter(parameterSpec);

    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                "the {@link " + ((ClassName) parameterSpec.type).simpleName() + "}");

    ParameterSpec configurationOverridesParameter = generateConfigurationOverridesParameter(withAnnotations);
    methodBuilder.addParameter(configurationOverridesParameter);
    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, configurationOverridesParameter.name, "the {@link ConfigurationOverrides}");

    ParameterSpec streamingHelperParameter = generateStreamingHelperParameter();
    methodBuilder.addParameter(streamingHelperParameter);
    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, streamingHelperParameter.name, "the {@link StreamingHelper}");

    if (requiresCallbackParameter()) {
      ParameterSpec completionCallbackParameter = generateCompletionCallbackParameter();
      methodBuilder.addParameter(completionCallbackParameter);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, completionCallbackParameter.name, "the operation's {@link CompletionCallback}");
    }
  }

  private AnnotationSpec getOutputResolverAnnotation() {
    AnnotationSpec.Builder builder = AnnotationSpec
        .builder(OutputResolver.class);

    if (outputMetadataResolver != null && outputMetadataResolver.getRequiresMetadataResolver()) {
      builder.addMember("output", "$T.class",
                        ClassName.get(outputMetadataResolver.getPackage(),
                                      outputMetadataResolver.getClassName()));
    } else if (muleOutputResolver.isPresent()) {
      ClassName className = MuleAnnotationsUtils.getClassName(muleOutputResolver.get());
      builder.addMember("output", "$T.class", className);
    }
    return builder
        .addMember("attributes", "$T.class", ClassName.get(SdkHttpResponseAttributesMetadataResolver.class))
        .build();
  }

  protected void addContentParameters(CodeBlock.Builder javaDoc, MethodSpec.Builder methodBuilder, boolean withAnnotations) {
    if (content != null) {
      ParameterSpec parameterSpec = content.generateContentParameter(withAnnotations);
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name, "the content to use");
    }
  }

  private Pagination getPagination() {
    return this.operation.getPagination();
  }

  private boolean isQueryParamDefinedInPagination(String paramName) {
    Pagination pagination = getPagination();
    if (pagination != null && pagination.getPaginationParameter() != null) {
      return pagination.getPaginationParameter().equalsIgnoreCase(paramName);
    }

    return false;
  }

  private boolean isQueryParamDefinedInMarkerPagination(String paramName) {
    return getPagination().getPaginationParameter().equals(paramName);
  }

  private boolean isEndpointMarkerPagination() {
    Pagination pagination = getPagination();
    return pagination != null && pagination.getType().equals(PaginationType.MARKER);
  }

  private AnnotationSpec generateDisplayNameAnnotation() {
    return AnnotationSpec.builder(DisplayName.class)
        .addMember(VALUE_MEMBER, "$S", operation.getDisplayName())
        .build();
  }

  private AnnotationSpec generateAliasAnnotation() {
    return AnnotationSpec
        .builder(Alias.class)
        .addMember(VALUE_MEMBER, "$S", operation.getAlias())
        .build();
  }

  private AnnotationSpec generateSidecarAnnotation() {
    return AnnotationSpec.builder(Ignore.class)
        .build();
  }

  public CodeBlock.Builder generateCommonOperationMethodBody(List<ParameterSpec> parameters) {
    final CodeBlock.Builder builder = CodeBlock.builder();
    generateCommonOperationMethodBody(builder, parameters);
    return builder;
  }

  public CodeBlock generateOperationLowerMethodBody(List<ParameterSpec> parameters) throws TemplatingException {
    final CodeBlock.Builder builder = CodeBlock.builder();
    builder.addStatement("super.$L($L)",
                         getJavaMethodName(),
                         parameters.stream().map(param -> param.name).collect(joining(", ")));
    return builder.build();
  }

  public CodeBlock generateOperationInterceptorMethodBody() throws TemplatingException {
    final CodeBlock.Builder builder = CodeBlock.builder();

    builder.add("// Add custom code");
    return builder.build();
  }

  public void generateCommonOperationMethodBody(CodeBlock.Builder methodBody, List<ParameterSpec> parameters) {
    String baseUriString = "connection.getBaseUri()";

    if (isNotBlank(operation.getAlternativeBaseUri()) && connectorModel.getBaseUri().isMultipleBaseUri()) {
      String multipleBaseUri =
          connectorModel.getBaseUri().getMultipleBaseUriOrDefault(operation.getAlternativeBaseUri());

      baseUriString = "\"" + multipleBaseUri + "\"";
    }

    if (operation.isAdapter()) {
      methodBody.add(generateInterceptorCall(parameters));
      methodBody.add(generateAuxParameterBindings());
    }

    generateRestRequestBuilderInitialization(methodBody, baseUriString);

    methodBody.add(".setQueryParamFormat($L)", QUERY_PARAM_FORMAT_FIELD);

    if (operation.getInputMetadata() != null && operation.getInputMetadata().getMediaType() != null) {

      String mediaType = operation.getInputMetadata().getMediaType().toString();

      if (operation.getInputMetadata().getMediaType().equals(MULTIPART_FORM_DATA_TYPE)) {
        mediaType = mediaType + "; boundary=__rc2_34b212";
      }

      methodBody.add(".$L($S, $S)",
                     ADD_HEADER_METHOD_NAME,
                     CONTENT_TYPE_HEADER_NAME,
                     mediaType);
    }

    if (operation.getOutputMetadata() != null && operation.getOutputMetadata().getMediaType() != null) {
      methodBody.add(".$L($S, $S)",
                     ADD_HEADER_METHOD_NAME,
                     ACCEPT_HEADER_NAME,
                     operation.getOutputMetadata().getMediaType().toString());
    }

    for (SdkParameter uriParameter : allUriParameters) {
      methodBody.add(generateRequestBuilderParameterCodeBlock(uriParameter,
                                                              ADD_URI_PARAM_METHOD_NAME,
                                                              ADD_MULTIPLE_URI_PARAM_METHOD_NAME));
    }

    for (SdkParameter queryParam : allQueryParameters) {
      if (!isQueryParamDefinedInPagination(queryParam.getExternalName())) {
        methodBody.add(generateRequestBuilderParameterCodeBlock(queryParam,
                                                                ADD_QUERY_PARAM_METHOD_NAME,
                                                                ADD_MULTIPLE_QUERY_PARAM_METHOD_NAME));
      }
    }

    for (SdkParameter header : allHeaders) {
      methodBody.add(generateRequestBuilderParameterCodeBlock(header,
                                                              ADD_HEADER_METHOD_NAME,
                                                              ADD_MULTIPLE_HEADER_METHOD_NAME));
    }

    addSetBodyMethod(methodBody);

    methodBody.add(";");
  }

  public CodeBlock generateInterceptorCall(List<ParameterSpec> parameters) {
    final CodeBlock.Builder builder = CodeBlock.builder();
    builder.addStatement("$L($L)",
                         INTERCEPTOR_METHOD_NAME,
                         parameters.stream().map(param -> param.name).collect(joining(", ")));
    return builder.build();
  }

  protected void generateRestRequestBuilderInitialization(CodeBlock.Builder methodBody, String baseUriString) {
    if (operation.isAdapter()) {
      methodBody.add("$T builder = $L($L, $L, $T.$L, parameters, overrides, connection, config)",
                     RestRequestBuilder.class,
                     GET_REQUEST_BUILDER_WITH_BINDINGS_METHOD,
                     baseUriString,
                     OPERATION_PATH_FIELD,
                     HttpConstants.Method.class,
                     operation.getHttpMethod().name().toUpperCase());
    } else {
      methodBody.add("$1T builder = new $1T($2L, $3L, $4T.$5L, parameters)",
                     RestRequestBuilder.class,
                     baseUriString,
                     OPERATION_PATH_FIELD,
                     HttpConstants.Method.class,
                     operation.getHttpMethod().name().toUpperCase());
    }
  }

  protected void addSetBodyMethod(CodeBlock.Builder methodBody) {
    if (content != null) {
      methodBody.add(".setBody($L, overrides.getStreamingType())", content.getContentParameterJavaName());
    }
  }

  private CodeBlock generateRequestBuilderParameterCodeBlock(SdkParameter parameter, String addSingleValueMethodName,
                                                             String addMultipleValueMethodName) {
    CodeBlock.Builder builder = CodeBlock.builder();

    String methodName = parameter.isArrayType() ? addMultipleValueMethodName : addSingleValueMethodName;

    builder.add(".$L($S, $L)", methodName, parameter.getExternalName(), getParameterValueStatement(parameter));

    return builder.build();
  }

  private CodeBlock getParameterValueStatement(SdkParameter parameter) {
    CodeBlock.Builder builder = CodeBlock.builder();

    if (parameter.isArrayType()) {
      CodeBlock getter = parameter.getInnerTypeStringValueGetter("v");
      if (getter.toString().equalsIgnoreCase("v")) {
        builder.add("$L.stream().filter($T::nonNull).collect($T.toList())",
                    parameter.getJavaName(),
                    Objects.class,
                    Collectors.class);
      } else {
        builder.add("$L.stream().filter($T::nonNull).map(v -> $L).collect($T.toList())",
                    parameter.getJavaName(),
                    Objects.class,
                    getter,
                    Collectors.class);
      }
    } else {
      builder.add("$L", parameter.getStringValueGetter());
    }

    return builder.build();
  }

  private AnnotationSpec generateThrowsAnnotation() {
    return AnnotationSpec
        .builder(Throws.class)
        .addMember(VALUE_MEMBER, "$T.class", RequestErrorTypeProvider.class)
        .build();
  }

  private AnnotationSpec generateMediaTypeAnnotation() {
    return AnnotationSpec
        .builder(MediaType.class)
        .addMember(VALUE_MEMBER, "$S", operation.getOutputMetadata().getMediaType())
        .build();
  }

  private AnnotationSpec generateSampleDataAnnotation() {
    AnnotationSpec.Builder annotationBuilder = AnnotationSpec
        .builder(SampleData.class)
        .addMember(VALUE_MEMBER, "$T.class",
                   ClassName.get(sampleDataProvider.getPackage(), sampleDataProvider.getJavaClassName()));
    if (operation.getSampleData() instanceof ResolverReference) {
      ResolverReference<SampleDataDefinition> sampleDataDefinitionReferenceResolver =
          (ResolverReference<SampleDataDefinition>) operation.getSampleData();
      addBindingAnnotation(annotationBuilder, sampleDataDefinitionReferenceResolver.getArguments());
    } else if (operation.getSampleData() instanceof SampleDataDefinition) {
      SampleDataDefinition sampleDataDefinition = (SampleDataDefinition) operation.getSampleData();
      HttpRequestDataExpression result = sampleDataDefinition.getResult();
      if (result.getHttpRequestBinding() != null) {
        List<Argument> arguments = Stream.of(result.getHttpRequestBinding().getHeader(),
                                             result.getHttpRequestBinding().getQueryParameter(),
                                             result.getHttpRequestBinding().getUriParameter())
            .flatMap(argumentList -> argumentList.stream())
            .collect(toList());
        addBindingAnnotation(annotationBuilder, arguments);
      }
    }
    return annotationBuilder.build();
  }

  private void addBindingAnnotation(AnnotationSpec.Builder annotationBuilder, List<Argument> arguments) {
    String bodyAccessor = BODY.getAccessorName() + ".";
    for (Argument argument : arguments) {
      if (argument.getValue().getValue().contains(bodyAccessor))
        annotationBuilder.addMember("bindings", "$L",
                                    AnnotationSpec.builder(Binding.class)
                                        .addMember("actingParameter", "$S", getActingParameterJavaName(argument))
                                        .addMember("extractionExpression", "$S",
                                                   transformBodyPrefix(argument.getValue().getValue(),
                                                                       content.getContentParameterJavaName()))
                                        .build());
    }
  }

  private AnnotationSpec generateDefaultMediaTypeAnnotation() {
    return AnnotationSpec
        .builder(MediaType.class)
        .addMember(VALUE_MEMBER, "$S", operationMethodRequiresBody() ? "application/json" : "text/plain")
        .build();
  }

  private ParameterSpec generateConfigParameter(boolean withAnnotations) {
    ParameterSpec.Builder parameterSpecBuilder = ParameterSpec.builder(RestConfiguration.class, "config");
    if (withAnnotations) {
      parameterSpecBuilder.addAnnotation(Config.class);
    }
    return parameterSpecBuilder.build();
  }

  private ParameterSpec generateConnectionParameter(boolean withAnnotations) {
    ParameterSpec.Builder parameterSpecBuilder = ParameterSpec.builder(RestConnection.class, "connection");
    if (withAnnotations) {
      parameterSpecBuilder.addAnnotation(Connection.class);
    }
    return parameterSpecBuilder.build();
  }

  private ParameterSpec generateRequestParametersParameter(boolean withAnnotations) {

    Class<?> requestParameterClass;
    if (operation.isRequestBodyBound() && operation.getInputMetadata() != null) {
      requestParameterClass = NonEntityRequestParameters.class;
    } else {
      requestParameterClass =
          operation.getInputMetadata() != null ? EntityRequestParameters.class : NonEntityRequestParameters.class;
    }

    ParameterSpec.Builder parameterSpecBuilder = ParameterSpec.builder(requestParameterClass, "parameters");

    if (withAnnotations) {
      AnnotationSpec parameterGroupAnnotation =
          AnnotationSpec
              .builder(ParameterGroup.class)
              .addMember(NAME_MEMBER, REQUEST_PARAMETERS_GROUP_NAME)
              .build();

      parameterSpecBuilder.addAnnotation(parameterGroupAnnotation);
    }

    return parameterSpecBuilder.build();
  }

  private ParameterSpec generateConfigurationOverridesParameter(boolean withAnnotations) {

    ParameterSpec.Builder parameterSpecBuilder = ParameterSpec.builder(ConfigurationOverrides.class, "overrides");

    if (withAnnotations) {
      AnnotationSpec parameterGroupAnnotation =
          AnnotationSpec
              .builder(ParameterGroup.class)
              .addMember(NAME_MEMBER, CONNECTOR_OVERRIDES)
              .build();

      parameterSpecBuilder.addAnnotation(parameterGroupAnnotation);
    }

    return parameterSpecBuilder.build();
  }

  private ParameterSpec generateStreamingHelperParameter() {
    return ParameterSpec
        .builder(StreamingHelper.class, "streamingHelper")
        .build();
  }

  private ParameterSpec generateCompletionCallbackParameter() {
    MessageOutputType messageOutputType = getMessageOutputType();
    return ParameterSpec
        .builder(ParameterizedTypeName.get(ClassName.get(CompletionCallback.class),
                                           messageOutputType.getOutputType(),
                                           messageOutputType.getAttributeOutputType()),
                 "callback")
        .build();
  }

  /**
   * Utility method to encapsulate how the "return" type (or the output of the current operation's types) is calculated
   */
  protected MessageOutputType getMessageOutputType() {
    if (!isVoidOperation() && (outputMetadataResolver != null || operationMethodRequiresBody())) {
      return new MessageOutputType(InputStream.class, Object.class);
    } else {
      return new MessageOutputType(String.class, Object.class);
    }
  }

  protected boolean operationMethodRequiresBody() {
    return operation.getHttpMethod().equals(HTTPMethod.GET)
        || operation.getHttpMethod().equals(HTTPMethod.POST)
        || operation.getHttpMethod().equals(HTTPMethod.PATCH)
        || operation.getHttpMethod().equals(HTTPMethod.OPTIONS);
  }

  protected boolean isVoidOperation() {
    return operation.getVoidOperation() != null && operation.getVoidOperation();
  }

  protected boolean requiresConnectionParameter() {
    return true;
  }

  protected boolean requiresCallbackParameter() {
    return true;
  }

  protected boolean requiresMediaTypeAnnotation() {
    return true;
  }

  /**
   * Returns all the parameters of this operation (URI + QUERY + HEADER)
   */
  public List<SdkParameter> getAllParameters() {
    final List<SdkParameter> allParameters = new ArrayList<>();
    allParameters.addAll(allUriParameters);
    allParameters.addAll(allQueryParameters);
    allParameters.addAll(allHeaders);
    allParameters.addAll(auxParameters);
    return allParameters;
  }

  public List<SdkField> getAllBodyFields() {
    return allBodyFields;
  }

  public ConnectorOperation getConnectorOperation() {
    return this.operation;
  }

  public static class MessageOutputType {

    private final TypeName outputType;
    private final TypeName attributeOutputType;

    public MessageOutputType(TypeName outputType, TypeName attributeOutputType) {
      this.outputType = outputType;
      this.attributeOutputType = attributeOutputType;
    }

    public MessageOutputType(Class<?> outputType, Class<?> attributeOutputType) {
      this(ClassName.get(outputType), ClassName.get(attributeOutputType));
    }

    public TypeName getOutputType() {
      return outputType;
    }

    public TypeName getAttributeOutputType() {
      return attributeOutputType;
    }
  }

  // Operation Adapters - IO Transformations
  private static final String GET_REQUEST_BINDINGS_METHOD = "getRequestBindings";
  private static final String GET_RESPONSE_BINDINGS_METHOD = "getResponseBindings";
  private static final String PARAMETER_BINDING_LOCAL_VARIABLE = "bindings";
  private static final String ADD_URI_BINDING_METHOD = "addUriParamBinding";
  private static final String ADD_QUERY_BINDING_METHOD = "addQueryParamBinding";
  private static final String ADD_HEADER_BINDING_METHOD = "addHeaderBinding";
  private static final String ADD_BODY_BINDING_METHOD = "setBody";
  private static final String ADD_PARAMETER_BINDING_METHOD = "addParameterToBindingContext";
  private static final String GET_REQUEST_BUILDER_WITH_BINDINGS_METHOD = "getRequestBuilderWithBindings";

  protected CodeBlock generateAuxParameterBindings() {
    CodeBlock.Builder builder = CodeBlock.builder();

    for (SdkParameter parameter : auxParameters) {
      builder.add("$L($S, $L);", ADD_PARAMETER_BINDING_METHOD, parameter.getExternalName(), parameter.getJavaName());
    }

    return builder.build();
  }

  private MethodSpec generateDoBeforeRequestMethod(List<ParameterSpec> parameters) {
    MethodSpec.Builder methodSpecBuilder =
        MethodSpec.methodBuilder(INTERCEPTOR_METHOD_NAME)
            .addModifiers(PROTECTED)
            .addModifiers(ABSTRACT);

    parameters.forEach(methodSpecBuilder::addParameter);
    return methodSpecBuilder.build();
  }

  private MethodSpec generateRequestBindingsMethod() {
    return generateRequestBindingMethod(GET_REQUEST_BINDINGS_METHOD,
                                        HttpRequestBinding.class,
                                        operation.getRequestBindings().orElse(null));
  }

  private MethodSpec generateResponseBindingsMethod() {
    return generateRequestBindingMethod(GET_RESPONSE_BINDINGS_METHOD,
                                        HttpResponseBinding.class,
                                        operation.getResponseBindings().orElse(null));
  }

  private MethodSpec generateRequestBindingMethod(String methodName,
                                                  Class<?> builderClass,
                                                  List<ParameterBinding> parameterBindings) {
    MethodSpec.Builder methodBuilder =
        MethodSpec.methodBuilder(methodName)
            .addModifiers(PROTECTED)
            .returns(builderClass)
            .addAnnotation(Override.class);

    methodBuilder.addStatement("$1T $2L = new $1T()",
                               builderClass,
                               PARAMETER_BINDING_LOCAL_VARIABLE);

    if (parameterBindings != null) {
      for (ParameterBinding binding : parameterBindings) {
        if (!binding.isIgnored()) {
          if (binding.getParameterType().equals(BODY)) {
            methodBuilder.addStatement("$L.$L($S)",
                                       PARAMETER_BINDING_LOCAL_VARIABLE,
                                       ADD_BODY_BINDING_METHOD,
                                       binding.getExpression());
          } else {
            methodBuilder.addStatement("$L.$L($S, $S)",
                                       PARAMETER_BINDING_LOCAL_VARIABLE,
                                       getParameterBindingAddMethodName(binding),
                                       binding.getName(),
                                       binding.getExpression());
          }
        }
      }
    }

    methodBuilder.addStatement("return $1L", PARAMETER_BINDING_LOCAL_VARIABLE);

    return methodBuilder.build();
  }

  private String getParameterBindingAddMethodName(ParameterBinding binding) {
    switch (binding.getParameterType()) {
      case QUERY:
        return ADD_QUERY_BINDING_METHOD;
      case HEADER:
        return ADD_HEADER_BINDING_METHOD;
      case URI:
        return ADD_URI_BINDING_METHOD;
    }

    throw new IllegalArgumentException("Parameter type not supported: " + binding.getParameterType());
  }
}
