/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.TRIGGER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;

import com.mulesoft.connectivity.rest.commons.api.source.RestPollingSource;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.TypeSchemaPool;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder.ParameterBuilder;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.TriggerParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.layers.SdkTriggerBaseLayer;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.layers.SdkTriggerImplementationLayer;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.layers.SdkTriggerInterceptorLayer;

import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeName;

public class SdkTrigger extends JavaTemplateEntity {

  public static final String CLASSNAME_POSTFIX = "Trigger";
  private static final String PACKAGE_POSTFIX = ".internal.source";

  private final SdkTriggerBaseLayer baseLayer;
  private final SdkTriggerInterceptorLayer interceptorLayer;
  private final SdkTriggerImplementationLayer implementationLayer;
  private final SdkConnector sdkConnector;
  private final Trigger trigger;
  private final List<SdkParameter> sdkParameters;
  private final String javaClassName;
  private final String packageName;

  public SdkTrigger(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, Trigger trigger,
                    RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);

    this.sdkConnector = sdkConnector;
    this.trigger = trigger;

    sdkParameters = buildSdkParameters();

    javaClassName = getJavaClassName();
    packageName = getPackage();

    baseLayer = new SdkTriggerBaseLayer(outputDir, connectorModel, sdkConnector, trigger, sdkParameters, javaClassName,
                                        packageName, ClassName.get(RestPollingSource.class),
                                        runConfiguration);
    interceptorLayer = new SdkTriggerInterceptorLayer(outputDir, connectorModel, sdkConnector, trigger, javaClassName,
                                                      packageName, baseLayer.getTypeName(),
                                                      runConfiguration);
    implementationLayer =
        new SdkTriggerImplementationLayer(outputDir, connectorModel, sdkConnector, trigger, javaClassName, packageName,
                                          interceptorLayer.getTypeName(), sdkParameters,
                                          runConfiguration);
  }

  public String getJavaClassName() {
    return getJavaUpperCamelNameFromXml(trigger.getInternalName()) + CLASSNAME_POSTFIX;
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + PACKAGE_POSTFIX;
  }

  private List<SdkParameter> buildSdkParameters() {
    return trigger.getParameters().stream()
        .map(this::buildSdkParameter)
        .collect(Collectors.toList());
  }

  private SdkParameter buildSdkParameter(TriggerParameter triggerParameter) {
    // TODO RSDK-620: [Tech Debt] Refactor parameter hierarchy, the rest sdk generatar should not instantiate classes of other
    // modules new Parameter(), it has to use the TriggerParameter
    String parameterIdentifier = triggerParameter.getExternalName();
    ParameterBuilder paramBuilder = new ParameterBuilder(TRIGGER, triggerParameter.getExternalName(), parameterIdentifier)
        .displayName(triggerParameter.getDisplayName())
        .description(triggerParameter.getDescription())
        .required(triggerParameter.isRequired());
    paramBuilder.getTypeDefinitionBuilder().primitive(getPrimitiveType(triggerParameter.getType()));

    return new SdkParameter(outputDir, connectorModel, sdkConnector, javaClassName,
                            paramBuilder.buildParameter(new TypeSchemaPool(), Optional.empty()), this,
                            runConfiguration);
  }

  private PrimitiveTypeDefinition.PrimitiveType getPrimitiveType(ParameterDataType parameterDataType) {
    switch (parameterDataType) {
      case NUMBER:
        return PrimitiveTypeDefinition.PrimitiveType.NUMBER;
      case INTEGER:
        return PrimitiveTypeDefinition.PrimitiveType.INTEGER;
      case LONG:
        return PrimitiveTypeDefinition.PrimitiveType.LONG;
      case STRING:
        return PrimitiveTypeDefinition.PrimitiveType.STRING;
      case BOOLEAN:
        return PrimitiveTypeDefinition.PrimitiveType.BOOLEAN;
      case LOCAL_DATE_TIME:
        return PrimitiveTypeDefinition.PrimitiveType.LOCAL_DATE_TIME;
      case ZONED_DATE_TIME:
        return PrimitiveTypeDefinition.PrimitiveType.ZONED_DATE_TIME;
      default:
        throw new IllegalArgumentException("Parameter type not supported. This is a bug.");
    }
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    baseLayer.applyTemplates();
    interceptorLayer.applyTemplates();
    implementationLayer.applyTemplates();
  }

  public TypeName getTypeNameForConfig() {
    return implementationLayer.getTypeName();
  }
}
