/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.TRIGGER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition.simplePrimitiveType;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static java.util.Collections.emptyList;
import static org.apache.commons.lang3.ObjectUtils.defaultIfNull;

import com.mulesoft.connectivity.rest.commons.api.source.RestPollingSource;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.generic.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.TriggerParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.templating.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.SdkConnector;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkAuxiliarParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.parameter.SdkParameter;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.layers.SdkTriggerBaseLayer;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.layers.SdkTriggerImplementationLayer;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.trigger.layers.SdkTriggerRefinementLayer;
import com.mulesoft.connectivity.rest.sdk.templating.sdk.valueprovider.SdkValueProvidable;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.TypeName;

public class SdkTrigger extends JavaTemplateEntity implements SdkTriggerTemplate, SdkValueProvidable {

  public static final String TRIGGER_CLASSNAME_SUFFIX = "Trigger";
  private static final String PACKAGE_SUFFIX = ".internal.source";

  private final SdkTriggerBaseLayer baseLayer;
  private final SdkTriggerRefinementLayer refinementLayer;
  private final SdkTriggerImplementationLayer implementationLayer;
  private final SdkConnector sdkConnector;
  private final Trigger trigger;
  private final List<SdkParameter> sdkParameters;
  private final String javaClassName;
  private final String packageName;

  public SdkTrigger(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector, Trigger trigger,
                    RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);

    this.sdkConnector = sdkConnector;
    this.trigger = trigger;

    sdkParameters = buildSdkParameters();

    javaClassName = getJavaClassName();
    packageName = getPackage();

    baseLayer = new SdkTriggerBaseLayer(outputDir, connectorModel, sdkConnector, trigger, sdkParameters, javaClassName,
                                        packageName, ClassName.get(RestPollingSource.class),
                                        runConfiguration);
    refinementLayer = new SdkTriggerRefinementLayer(outputDir, connectorModel, sdkConnector, trigger, javaClassName,
                                                    packageName, baseLayer.getTypeName(),
                                                    runConfiguration);
    implementationLayer =
        new SdkTriggerImplementationLayer(outputDir, connectorModel, sdkConnector, trigger, javaClassName, packageName,
                                          refinementLayer.getTypeName(), sdkParameters,
                                          runConfiguration);
  }

  @Override
  public String getJavaClassName() {
    return getJavaUpperCamelNameFromXml(trigger.getInternalName()) + TRIGGER_CLASSNAME_SUFFIX;
  }

  @Override
  public List<SdkParameter> getAllParameters() {
    return sdkParameters;
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + PACKAGE_SUFFIX;
  }

  private List<SdkParameter> buildSdkParameters() throws TemplatingException {
    final List<SdkParameter> list = new ArrayList<>();
    for (TriggerParameter parameter : trigger.getParameters()) {
      list.add(buildSdkParameter(parameter));
    }
    return list;
  }

  private SdkAuxiliarParameter buildSdkParameter(TriggerParameter triggerParameter) throws TemplatingException {
    // TODO RSDK-620: [Tech Debt] Refactor parameter hierarchy, the rest sdk generatar should not instantiate classes of other
    // modules new Parameter(), it has to use the TriggerParameter

    TypeDefinition typeDefinition = triggerParameter.getTypeDefinition();
    if (typeDefinition == null && triggerParameter.getMuleTypeResolver() == null) {
      typeDefinition = getTypeDefinition(triggerParameter.getType());
    }

    Parameter parameter = new Parameter(triggerParameter.getDisplayName(),
                                        triggerParameter.getExternalName(),
                                        triggerParameter.getExternalName(),
                                        TRIGGER,
                                        typeDefinition,
                                        triggerParameter.getMuleAlias(),
                                        triggerParameter.getDescription(),
                                        triggerParameter.isRequired() == null || triggerParameter.isRequired(),
                                        null,
                                        false,
                                        triggerParameter.getValueProvider(),
                                        triggerParameter.getMuleMetadataKeyId(),
                                        triggerParameter.getMuleTypeResolver(),
                                        triggerParameter.getMuleContent(),
                                        emptyList(),
                                        triggerParameter.getSummary());

    return new SdkAuxiliarParameter(outputDir, connectorModel, sdkConnector, trigger.getInternalName(), javaClassName,
                                    parameter, this,
                                    runConfiguration, defaultIfNull(triggerParameter.getMuleContent(), false));
  }

  private TypeDefinition getTypeDefinition(ParameterDataType parameterDataType) {
    PrimitiveTypeDefinition.PrimitiveType primitiveType;

    switch (parameterDataType) {
      case NUMBER:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.NUMBER;
        break;
      case INTEGER:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.INTEGER;
        break;
      case LONG:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.LONG;
        break;
      case STRING:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.STRING;
        break;
      case BOOLEAN:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.BOOLEAN;
        break;
      case LOCAL_DATE_TIME:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.LOCAL_DATE_TIME;
        break;
      case ZONED_DATE_TIME:
        primitiveType = PrimitiveTypeDefinition.PrimitiveType.ZONED_DATE_TIME;
        break;
      default:
        throw new IllegalArgumentException("Parameter type not supported. This is a bug.");
    }

    return simplePrimitiveType(primitiveType);
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    baseLayer.applyTemplates();
    refinementLayer.applyTemplates();
    implementationLayer.applyTemplates();

    for (SdkParameter sdkParameter : sdkParameters) {
      sdkParameter.applyTemplates();
    }
  }

  @Override
  public TypeName getTypeNameForConfig() {
    return implementationLayer.getTypeName();
  }
}
