/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.mojo;

import static java.lang.String.format;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.Parameter;
import org.json.JSONObject;
import org.skyscreamer.jsonassert.JSONParser;

public abstract class BaseRestSdkMojo extends AbstractMojo {

  public static final String EXCHANGE_JSON_FILE = "exchange.json";

  @Parameter(property = "apiSpec", required = true)
  protected File apiSpec;

  @Parameter(property = "outputDir", defaultValue = ".")
  protected File outputDir;

  @Parameter(property = "skipValidation", defaultValue = "false")
  protected boolean skipValidation;

  /**
   * Determines if the apiSpec is an exchange file, if so it will assume there's a main file to properly point out the API spec
   * relative to the exchange.json file. If it fails reading the main element, then it will rollback to assuming the apiSpec is
   * actually an API file.
   *
   * @return the file to an API spec
   * @throws MojoExecutionException if the exchange.json file cannot be parsed
   */
  public File getApiSpec(Log log) throws MojoExecutionException {

    File result = apiSpec;

    if (apiSpec.getName().equals(EXCHANGE_JSON_FILE)) {
      try {
        log.info(String.format("It seems it's an exchange dependency, reading the [%s] file to extract the main api file",
                               apiSpec.getAbsolutePath()));
        String content = new String(Files.readAllBytes(Paths.get(result.toPath().toString())));
        Object obj = JSONParser.parseJSON(content);
        JSONObject jsonObject = (JSONObject) obj;
        String mainFile = (String) jsonObject.get("main");
        if (StringUtils.isNotBlank(mainFile)) {
          log.info(String.format("Main file from exchange is [%s]", mainFile));
          File apiSpecFileFromExchange = new File(result.getParent(), mainFile);
          log.info(String.format("Calculated api file [%s]", apiSpecFileFromExchange));
          result = apiSpecFileFromExchange;
        } else {
          log.info(String
              .format("The file [%s] wasn't an exchange file after all, defaulting to assuming is an OAS json file (either OAS 2 or 3)",
                      apiSpec.getAbsolutePath()));
        }
      } catch (IOException e) {
        throw new MojoExecutionException(format("There has been an error reading the [%s] ", apiSpec), e);
      }
    }
    return result;
  }
}
