/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.mojo;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.json.JSONObject;
import org.skyscreamer.jsonassert.JSONParser;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.List;
import java.util.Map;

import static org.apache.maven.plugins.annotations.LifecyclePhase.GENERATE_SOURCES;
import static java.lang.String.format;
import static org.apache.maven.plugins.annotations.ResolutionScope.COMPILE_PLUS_RUNTIME;

@Mojo(name = "generateSources", defaultPhase = GENERATE_SOURCES,
    requiresDependencyCollection = COMPILE_PLUS_RUNTIME,
    requiresDependencyResolution = COMPILE_PLUS_RUNTIME)
public class GenerateSourcesConnectorMojo extends BaseRestSdkMojo {

  public static final String REST_SDK_GENERATED_SCHEMAS = "rest_sdk_generated_schemas";

  private static final String GENERATE_SOURCES_MOJO_PREVIOUS_RUN_PLACEHOLDER = "GENERATE_SOURCES_MOJO_PREVIOUS_RUN_PLACEHOLDER";

  @Parameter(property = "descriptor", required = true)
  private List<File> descriptor;

  @Parameter(property = "dynamicMetadata")
  private boolean dynamicMetadata;

  @Parameter(property = "overrideSourceDir", defaultValue = "src/main/override")
  private String overrideSourceDir;

  @Parameter(property = "refinementSourceDir", defaultValue = "src/main/refinement")
  private String refinementSourceDir;

  @Parameter(property = "generatedSourceDir", defaultValue = "target/generated-sources")
  private String generatedSourceDir;

  @Parameter(property = "generatedResourcesDir", defaultValue = "target/generated-resources")
  private String generatedResourcesDir;

  @Parameter(property = "ignoredDir", defaultValue = "target/ignored")
  private String ignoredDir;

  @Parameter(property = "generatedSchemasDir", defaultValue = REST_SDK_GENERATED_SCHEMAS)
  private String generatedSchemasDir;

  /**
   * @since 1.0
   */
  @Parameter(readonly = true, defaultValue = "${project}")
  private MavenProject project;

  public void execute() throws MojoExecutionException {
    if (hasExecutedBefore()) {
      getLog().info("Skipping execution of generateSources because it has already been run.");
      return;
    }

    // Validate connector Mojo
    this.validate();

    RestSdkExecutor.generateConnectorSources(getApiSpec(getLog()),
                                             descriptor,
                                             outputDir,
                                             skipValidation,
                                             dynamicMetadata,
                                             true,
                                             generatedSourceDir,
                                             generatedResourcesDir,
                                             overrideSourceDir,
                                             refinementSourceDir,
                                             ignoredDir,
                                             generatedSchemasDir);

    // Add generated sources to source folder
    this.project.addCompileSourceRoot(outputDir.toPath().resolve(overrideSourceDir).toFile().getPath());
    this.project.addCompileSourceRoot(outputDir.toPath().resolve(refinementSourceDir).toFile().getPath());
    this.project.addCompileSourceRoot(outputDir.toPath().resolve(generatedSourceDir).toFile().getPath());
    Resource resource = new Resource();
    resource.setDirectory(outputDir.toPath().resolve(generatedResourcesDir).toFile().getPath());
    this.project.addResource(resource);
  }

  /**
   * This method avoids running generation more than once. This could happen because other phases such as the maven-source-plugin,
   * can trigger the "generate-sources" phase since it needs its information.
   *
   * @return true if the MOJO run has already happened before
   */
  @SuppressWarnings("unchecked")
  protected boolean hasExecutedBefore() {
    Map<String, String> pluginContext = getPluginContext();
    if (pluginContext.containsKey(GENERATE_SOURCES_MOJO_PREVIOUS_RUN_PLACEHOLDER)) {
      return true;
    }
    getPluginContext().put(GENERATE_SOURCES_MOJO_PREVIOUS_RUN_PLACEHOLDER, GENERATE_SOURCES_MOJO_PREVIOUS_RUN_PLACEHOLDER);
    return false;
  }

  private void validate() throws MojoExecutionException {
    // Run validations
    validateSchemaFolderNonRepeatable();
  }

  private void validateSchemaFolderNonRepeatable() throws MojoExecutionException {
    for (Resource resource : project.getResources()) {
      String directory = resource.getDirectory();

      File file = new File(directory, REST_SDK_GENERATED_SCHEMAS);
      if (file.exists()) {
        throw new MojoExecutionException(format("A folder named %s already exists. REST SDK can not generate schemas there. "
            + "Configure properly the schemas output folder",
                                                generatedResourcesDir + "/" + REST_SDK_GENERATED_SCHEMAS));
      }
    }
  }

}
