/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.mojo;

import static com.mulesoft.connectivity.rest.sdk.api.DescriptorScaffolder.scaffoldDescriptor;
import static com.mulesoft.connectivity.rest.sdk.templating.api.ConnectorType.SDK_CONNECTOR;
import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.api.DescriptorMode;
import com.mulesoft.connectivity.rest.sdk.api.RestSdk;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;

import java.io.File;
import java.util.List;

import org.apache.maven.plugin.MojoExecutionException;
import org.slf4j.Logger;

public class RestSdkExecutor {

  private final static Logger LOGGER = getLogger(RestSdkExecutor.class);

  public static void createDescriptor(File apiSpec,
                                      String specFormat,
                                      File outputDir,
                                      String descriptorMode,
                                      boolean skipValidation)
      throws MojoExecutionException {

    try {
      scaffoldDescriptor(apiSpec,
                         isNotBlank(specFormat) ? SpecFormat.getFromString(specFormat) : null,
                         outputDir.toPath(),
                         DescriptorMode.getFromString(descriptorMode),
                         skipValidation);

      LOGGER.info("Descriptor written to " + outputDir.getAbsolutePath());
    } catch (Exception e) {
      String message = format("Failed to create descriptor, message [%s]", e.getMessage());
      LOGGER.debug(message, e);
      throw new MojoExecutionException(message, e);
    }
  }

  public static void createConnector(File apiSpec,
                                     List<File> descriptors,
                                     String specFormat,
                                     File outputDir,
                                     boolean skipValidation,
                                     boolean dynamicMetadata)
      throws MojoExecutionException {

    try {
      RestSdkRunConfiguration runConfiguration = RestSdkRunConfiguration.builder()
          .skipValidations(skipValidation)
          .useDynamicTypeResolvers(dynamicMetadata)
          .build();

      RestSdk restSdk = new RestSdk(apiSpec, apiSpec.getParent(), runConfiguration)
          .format(isNotBlank(specFormat) ? SpecFormat.getFromString(specFormat) : null);

      for (File descriptor : descriptors) {
        restSdk.descriptor(descriptor);
      }

      restSdk.generateConnector(SDK_CONNECTOR, outputDir.toPath(), runConfiguration);

      LOGGER.info("Connector written to " + outputDir.getAbsolutePath());
    } catch (Exception e) {
      String message = format("Failed to create connector, message [%s]", e.getMessage());
      LOGGER.debug(message, e);
      throw new MojoExecutionException(message, e);
    }
  }

}
