/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.mojo;

import static com.mulesoft.connectivity.rest.sdk.api.DescriptorScaffolder.scaffoldDescriptor;
import static com.mulesoft.connectivity.rest.sdk.mojo.GenerateSourcesConnectorMojo.REST_SDK_GENERATED_SCHEMAS;
import static java.lang.String.format;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.api.DescriptorMode;
import com.mulesoft.connectivity.rest.sdk.api.RestSdk;
import com.mulesoft.connectivity.rest.sdk.templating.api.RestSdkRunConfiguration;

import java.io.File;
import java.util.List;

import org.apache.maven.plugin.MojoExecutionException;
import org.slf4j.Logger;

public class RestSdkExecutor {

  private final static Logger LOGGER = getLogger(RestSdkExecutor.class);

  public static final String SRC_MAIN_JAVA = "src/main/java";
  public static final String SRC_MAIN_RESOURCES = "src/main/resources";

  public static void createDescriptor(File apiSpec,
                                      File outputDir,
                                      String descriptorMode,
                                      boolean skipValidation)
      throws MojoExecutionException {

    try {
      scaffoldDescriptor(apiSpec,
                         outputDir.toPath(),
                         DescriptorMode.getFromString(descriptorMode),
                         skipValidation);

      LOGGER.info("Descriptor written to " + outputDir.getAbsolutePath());
    } catch (Exception e) {
      String message = format("Failed to create descriptor, message [%s]", e.getMessage());
      LOGGER.debug(message, e);
      throw new MojoExecutionException(message, e);
    }
  }

  public static void createConnector(File apiSpec,
                                     List<File> descriptors,
                                     File outputDir,
                                     boolean skipValidation,
                                     boolean dynamicMetadata)
      throws MojoExecutionException {
    generateConnectorSources(apiSpec,
                             descriptors,
                             outputDir,
                             skipValidation,
                             dynamicMetadata,
                             false,
                             SRC_MAIN_JAVA,
                             SRC_MAIN_RESOURCES,
                             null,
                             null,
                             null,
                             REST_SDK_GENERATED_SCHEMAS);
  }

  public static void generateConnectorSources(File apiSpec,
                                              List<File> descriptors,
                                              File outputDir,
                                              boolean skipValidation,
                                              boolean dynamicMetadata,
                                              boolean regenerateMode,
                                              String generatedSourceDir,
                                              String generatedResourcesDir,
                                              String overrideSourceDir,
                                              String refinementSourceDir,
                                              String ignoredDir,
                                              String generatedSchemasDir)
      throws MojoExecutionException {

    try {
      RestSdkRunConfiguration runConfiguration = RestSdkRunConfiguration.builder()
          .skipValidations(skipValidation)
          .useDynamicTypeResolvers(dynamicMetadata)
          .generatedSourceDir(generatedSourceDir)
          .generatedResourcesDir(generatedResourcesDir)
          .overrideSourceDir(overrideSourceDir)
          .ignoredDir(ignoredDir)
          .regenerateMode(regenerateMode)
          .generatedSchemasDir(generatedSchemasDir)
          .build();

      RestSdk restSdk = new RestSdk(apiSpec, apiSpec.getParent(), runConfiguration);

      for (File descriptor : descriptors) {
        restSdk.descriptor(descriptor);
      }

      restSdk.generateConnectorSources(outputDir.toPath(), runConfiguration);

      runConfiguration.messageCollector().messages().forEach((message) -> {
        switch (message.getKind()) {
          case WARNING:
            LOGGER.warn(message.getMessage());
            break;
          case ERROR:
            LOGGER.error(message.getMessage());
            break;
          case INFO:
            LOGGER.info(message.getMessage());
            break;
        }
      });

      LOGGER.info("Connector written to " + outputDir.getAbsolutePath());
    } catch (NullPointerException npe) {
      String message =
          format("Failed to create connector with a NullPointerException, this is a Rest SDK bug please report it with a maven run with the '-e' (error) flag.");
      LOGGER.debug(message, npe);
      throw new MojoExecutionException(message, npe);
    } catch (Exception e) {
      String message = format("Failed to create connector, message [%s]", e.getMessage());
      LOGGER.debug(message, e);
      throw new MojoExecutionException(message, e);
    }
  }

}
