/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.mojo;

import org.apache.maven.model.Resource;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;

import java.io.File;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.mojo.RestSdkExecutor.createConnector;
import static org.apache.maven.plugins.annotations.LifecyclePhase.GENERATE_SOURCES;
import static java.lang.String.format;

@Mojo(name = "generateSources", defaultPhase = GENERATE_SOURCES,
    requiresDependencyCollection = ResolutionScope.COMPILE_PLUS_RUNTIME,
    requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class GenerateSourcesConnectorMojo extends BaseRestSdkMojo {

  public static final String REST_SDK_GENERATED_SCHEMAS = "rest_sdk_generated_schemas";

  @Parameter(property = "descriptor", required = true)
  private List<File> descriptor;

  @Parameter(property = "dynamicMetadata")
  private boolean dynamicMetadata;

  @Parameter(property = "overrideSourceDir", defaultValue = "src/main/override")
  private String overrideSourceDir;

  @Parameter(property = "generatedSourceDir", defaultValue = "target/generated-sources")
  private String generatedSourceDir;

  @Parameter(property = "generatedResourcesDir", defaultValue = "target/generated-resources")
  private String generatedResourcesDir;

  @Parameter(property = "ignoredDir", defaultValue = "target/ignored")
  private String ignoredDir;

  @Parameter(property = "generatedSchemasDir", defaultValue = REST_SDK_GENERATED_SCHEMAS)
  private String generatedSchemasDir;

  /**
   * @since 1.0
   */
  @Parameter(readonly = true, defaultValue = "${project}")
  private MavenProject project;

  public void execute() throws MojoExecutionException {

    // Validate connector Mojo
    this.validate();

    createConnector(apiSpec,
                    descriptor,
                    outputDir,
                    skipValidation,
                    dynamicMetadata,
                    true,
                    generatedSourceDir,
                    generatedResourcesDir,
                    overrideSourceDir,
                    ignoredDir,
                    generatedSchemasDir);

    // Add generated sources to source folder
    this.project.addCompileSourceRoot(outputDir.toPath().resolve(overrideSourceDir).toFile().getPath());
    this.project.addCompileSourceRoot(outputDir.toPath().resolve(generatedSourceDir).toFile().getPath());
    Resource resource = new Resource();
    resource.setDirectory(outputDir.toPath().resolve(generatedResourcesDir).toFile().getPath());
    this.project.addResource(resource);
  }

  private void validate() throws MojoExecutionException {
    // Run validations
    validateSchemaFolderNonRepeatable();
  }

  private void validateSchemaFolderNonRepeatable() throws MojoExecutionException {
    for (Resource resource : project.getResources()) {
      String directory = resource.getDirectory();

      File file = new File(directory, REST_SDK_GENERATED_SCHEMAS);
      if (file.exists()) {
        throw new MojoExecutionException(format("A folder named %s already exists. REST SDK can not generate schemas there. "
            + "Configure properly the schemas output folder",
                                                generatedResourcesDir + "/" + REST_SDK_GENERATED_SCHEMAS));
      }
    }
  }

}
