/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package com.mulesoft.connectivity.rest.sdk.mojo;

import static java.lang.String.format;
import static java.util.stream.Collectors.toList;
import static org.apache.maven.plugins.annotations.LifecyclePhase.COMPILE;

import org.mule.runtime.api.deployment.meta.MulePluginModel;
import org.mule.runtime.api.deployment.persistence.MulePluginModelJsonSerializer;

import com.mulesoft.connectivity.rest.commons.api.operation.BaseRestOperation;

import java.io.File;
import java.io.FileReader;
import java.net.URL;
import java.nio.file.Paths;
import java.util.List;

import org.apache.logging.log4j.core.util.IOUtils;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.reflections.Reflections;
import org.reflections.Store;
import org.reflections.scanners.AbstractScanner;
import org.reflections.util.ConfigurationBuilder;
import org.reflections.vfs.Vfs;

@Mojo(name = "validateConnector", defaultPhase = COMPILE)
public class ValidateConnectorMojo extends AbstractMojo {

  @Parameter(readonly = true, defaultValue = "${project}")
  private MavenProject project;

  @Parameter(property = "skipValidation", defaultValue = "false")
  protected boolean skipValidation;

  private static final String META_INF = "META-INF";
  private static final String MULE_ARTIFACT_JSON = "mule-artifact.json";
  private static final String MULE_ARTIFACT = "mule-artifact";

  private static final String AUTO_GENERATED_MULE_ARTIFACT_DESCRIPTOR =
      Paths.get(META_INF, "auto-generated-" + MULE_ARTIFACT_JSON).toString();
  private static final String MULE_ARTIFACT_DESCRIPTOR =
      Paths.get(META_INF, MULE_ARTIFACT, MULE_ARTIFACT_JSON).toString();

  public void execute() throws MojoExecutionException {
    if (skipValidation) {
      getLog().warn(format("Skipping validation, '%s' may be invalid", MULE_ARTIFACT_DESCRIPTOR));
      return;
    }
    String outputDirectory = project.getBuild().getOutputDirectory();
    File muleArtifactDescriptorFile = new File(outputDirectory, MULE_ARTIFACT_DESCRIPTOR);
    if (!muleArtifactDescriptorFile.exists()) {
      muleArtifactDescriptorFile = new File(outputDirectory, AUTO_GENERATED_MULE_ARTIFACT_DESCRIPTOR);
      getLog().info(format("Using '%s' to validate due to '%s' has not been created yet",
                           AUTO_GENERATED_MULE_ARTIFACT_DESCRIPTOR, MULE_ARTIFACT_DESCRIPTOR));
    }
    if (!muleArtifactDescriptorFile.exists()) {
      throw new MojoExecutionException(format("Descriptor file is not present at '%s' or '%s" +
          ", it should have been generated by the mule-extensions-maven-plugin. This is probably a bug",
                                              MULE_ARTIFACT_DESCRIPTOR, AUTO_GENERATED_MULE_ARTIFACT_DESCRIPTOR));
    }
    MulePluginModel pluginModel;
    try (FileReader muleArtifactDescriptorFileReader = new FileReader(muleArtifactDescriptorFile)) {
      pluginModel =
          new MulePluginModelJsonSerializer().deserialize(IOUtils.toString(muleArtifactDescriptorFileReader));
    } catch (Exception e) {
      throw new MojoExecutionException(format("Failed to validate connector's '%s' descriptor, message [%s]", MULE_ARTIFACT_JSON,
                                              e.getMessage()),
                                       e);
    }
    List<String> commonsRestSdkPackages = discoverCommonsRestSdkPackages();
    List<String> exportedPackages =
        (List<String>) pluginModel.getClassLoaderModelLoaderDescriptor().getAttributes().get("exportedPackages");
    List<String> commonsRestSdkExportedPackages = exportedPackages.stream()
        .filter(exportedPackage -> commonsRestSdkPackages.contains(exportedPackage))
        .collect(toList());
    if (!commonsRestSdkExportedPackages.isEmpty()) {
      throw new MojoExecutionException(format("Connector generated defines exported packages from RSDK commons library: %s",
                                              commonsRestSdkExportedPackages));
    }
  }

  private List<String> discoverCommonsRestSdkPackages() throws MojoExecutionException {
    URL url = BaseRestOperation.class.getProtectionDomain().getCodeSource().getLocation();
    try {
      List<String> packages = getPackages(url);
      getLog().debug(format("Packages discovered for URL: %s are: %s", url, packages));
      return packages;
    } catch (Exception e) {
      getLog().error(format("There was an error while scanning type elements from [%s]", url), e);
      throw new MojoExecutionException(
                                       "Failed to validate connector's due to packages for RSDK commons library couldn't be discovered",
                                       e);
    }
  }

  private List<String> getPackages(URL url) {
    Reflections reflections = new Reflections(new ConfigurationBuilder()
        .setUrls(url)
        .setScanners(new PackageScanner()));
    return reflections.getStore().values(PackageScanner.class.getSimpleName())
        .stream()
        .distinct()
        .sorted()
        .collect(toList());
  }

  /**
   * Custom scanner to resolve packages for each class found.
   */
  private class PackageScanner extends AbstractScanner {

    public boolean acceptsInput(String file) {
      return file.endsWith(".class");
    }

    @Override
    public Object scan(Vfs.File file, Object classObject, Store store) {
      Object resolvedClassObject = super.scan(file, classObject, store);
      put(store, getMetadataAdapter().getClassName(resolvedClassObject),
          new File(file.getRelativePath()).getParent().replace("/", "."));
      return resolvedClassObject;
    }

    public void scan(Object cls, Store store) {}

  }

}
