/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIType.MULTIPART;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA_TYPE;

import amf.client.model.domain.AnyShape;
import amf.client.model.domain.ArrayShape;
import amf.client.model.domain.DataNode;
import amf.client.model.domain.FileShape;
import amf.client.model.domain.NilShape;
import amf.client.model.domain.NodeShape;
import amf.client.model.domain.PropertyShape;
import amf.client.model.domain.ScalarNode;
import amf.client.model.domain.ScalarShape;
import amf.client.model.domain.SchemaShape;
import amf.client.model.domain.Shape;
import amf.client.model.domain.UnionShape;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveTypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;

import javax.ws.rs.core.MediaType;
import java.util.ArrayList;
import java.util.List;

public class AMFTypeModel extends APITypeModel {

  private AnyShape shape;

  public AMFTypeModel(APIPrimitiveTypeModel primitiveTypeModel) {
    super(primitiveTypeModel);
  }

  public AMFTypeModel(AnyShape anyShape, String mediaType) {
    this.shape = anyShape;

    this.example = buildExample();
    this.enumValues = buildEnumValues();
    this.mediaType = getMediaTypeForStringOrNull(mediaType);
    this.APITypeSchemaModel = AMFTypeSchemaModel.builder().build(anyShape, this.mediaType);
    this.apiType = buildTypeDefinitionClass(this.mediaType);
    this.displayName = buildDisplayName();
    this.description = buildDescription();
  }

  private String buildDescription() {
    return shape.description().isNullOrEmpty() ? null : shape.description().value();
  }

  private String buildDisplayName() {
    return shape.displayName().isNullOrEmpty() ? null : shape.displayName().value();
  }

  private APIType buildTypeDefinitionClass(MediaType mediaType) {
    if (shape instanceof ScalarShape || shape instanceof FileShape) {
      this.primitiveTypeModel = new AMFPrimitiveTypeModel(shape);
      return APIType.PRIMITIVE;
    }

    else if (MULTIPART_FORM_DATA_TYPE.equals(mediaType)
        && shape instanceof NodeShape
        && !((NodeShape) shape).properties().isEmpty()) {
      return MULTIPART;
    }

    else if (shape instanceof NodeShape || shape instanceof SchemaShape) {
      return APIType.OBJECT_TYPE;
    }

    else if (shape instanceof UnionShape) {
      return APIType.UNION_TYPE;
    }

    else if (shape instanceof ArrayShape) {
      return APIType.ARRAY;
    }

    else if (!shape.and().isEmpty() || !shape.or().isEmpty() || !shape.xone().isEmpty()) {
      return APIType.OBJECT_TYPE;
    }

    else if (shape instanceof NilShape) {
      //TODO marker: check if this makes sense, or delete it
      return null;
    }

    // This is for the case that the type is not defined (for instance, when there is an example but not a definition).
    else {
      return APIType.EMPTY;
    }
  }

  private String buildExample() {
    return !shape.examples().isEmpty() && shape.examples().get(0).value().nonEmpty()
        ? shape.examples().get(0).value().value() : null;
  }

  private List<String> buildEnumValues() {
    return (!shape.values().isEmpty())
        ? shape.values().stream().map(this::getEnumValue).collect(toList()) : null;
  }

  @Override
  public List<APIParameterModel> getParts() {
    if (apiType.equals(MULTIPART)) {
      List<APIParameterModel> parts = new ArrayList<>();

      for (PropertyShape s : ((NodeShape) shape).properties()) {
        parts.add(new AMFParameterModel(s, APIParameterType.PART));
      }

      return parts;
    } else {
      return emptyList();
    }
  }

  @Override
  public APITypeModel getInnerType() {
    if (shape instanceof ArrayShape) {
      Shape innerShape = ((ArrayShape) shape).items();

      if (innerShape instanceof AnyShape) {
        return new AMFTypeModel((AnyShape) innerShape, null);
      } else {
        //Defaulting to string type for shapes that dont implement AnyShape.
        //These shapes can not generate a json/xml schema.
        //TODO: RSDK-8: Recursive shapes can't generate schemas.
        return new AMFTypeModel(new AMFPrimitiveTypeModel(APIPrimitiveType.STRING));
      }
    } else {
      return null;
    }
  }

  @Override
  public List<APITypeModel> getUnionTypes() {
    if (shape instanceof UnionShape) {
      List<APITypeModel> list = new ArrayList<>();

      for (Shape x : ((UnionShape) shape).anyOf()) {
        AMFTypeModel amfTypeParser = new AMFTypeModel((AnyShape) x, null);
        list.add(amfTypeParser);
      }

      return list;
    } else {
      return emptyList();
    }
  }

  private String getEnumValue(DataNode x) {
    if (x instanceof ScalarNode) {
      return ((ScalarNode) x).value().value();
    }
    throw new IllegalArgumentException("Enum type not supported");
  }
}
