/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat.JSON_OAS;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat.JSON_OAS3;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat.RAML;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat.YAML_OAS;
import static java.lang.Thread.currentThread;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.InvalidSourceException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.resourceloader.AMFExchangeDependencyResourceLoader;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import amf.ProfileName;
import amf.ProfileNames;
import amf.client.AMF;
import amf.client.environment.DefaultEnvironment;
import amf.client.environment.Environment;
import amf.client.model.StrField;
import amf.client.model.document.BaseUnit;
import amf.client.model.document.Document;
import amf.client.model.domain.EndPoint;
import amf.client.model.domain.Server;
import amf.client.model.domain.WebApi;
import amf.client.parse.Oas20Parser;
import amf.client.parse.Oas20YamlParser;
import amf.client.parse.Oas30Parser;
import amf.client.parse.Oas30YamlParser;
import amf.client.parse.Parser;
import amf.client.parse.Raml10Parser;
import amf.client.resolve.Oas20Resolver;
import amf.client.resolve.Oas30Resolver;
import amf.client.resolve.Raml10Resolver;
import amf.client.validate.ValidationReport;
import org.slf4j.Logger;

public class AMFAPIModel extends APIModel {

  private final static Logger LOGGER = getLogger(AMFAPIModel.class);
  private final SpecFormat specFormat;
  private final WebApi webApi;

  public AMFAPIModel(File apiSpec, String rootDir, SpecFormat specFormat, boolean skipValidations)
      throws ModelGenerationException {
    initializeAMF();
    this.specFormat = specFormat;

    this.webApi = getWebApi(getParser(specFormat, rootDir), apiSpec.toPath(), skipValidations);

    this.apiName = buildApiName();
    this.description = buildDescription();
    this.protocols = buildProtocols();
    this.apiVersion = buildApiVersion();
    this.baseUri = buildBaseUri();
    this.operationsModel = buildOperationsModel();
  }

  private void initializeAMF() {
    try {
      AMF.init().get();
    } catch (ExecutionException e) {
      throw new RuntimeException("Could not register dialect.", e);
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new RuntimeException("Could not register dialect.", e);
    }
  }

  private String buildApiName() {
    return webApi.name().value();
  }

  private String buildDescription() {
    return webApi.description().nonEmpty() ? webApi.description().value() : EMPTY;
  }

  private String buildApiVersion() {
    return webApi.version() != null ? webApi.version().value() : null;
  }

  private String buildBaseUri() {
    String baseUri = null;

    Server server = webApi.servers().stream().filter(x -> x.url().nonEmpty()).findFirst().orElse(null);
    if (server != null) {
      baseUri = server.url().value();
    }

    return baseUri;
  }

  private List<String> buildProtocols() {
    return webApi.schemes().stream().map(StrField::value).collect(toList());
  }

  private List<APIOperationModel> buildOperationsModel() throws ModelGenerationException {
    List<APIOperationModel> operationsModel = new ArrayList<>();

    for (EndPoint endPoint : webApi.endPoints()) {
      for (amf.client.model.domain.Operation operation : endPoint.operations()) {
        operationsModel.add(new AMFOperationModel(endPoint, operation));
      }
    }

    return operationsModel;
  }

  private Parser getParser(SpecFormat format, String rootDir) {
    final Environment env = DefaultEnvironment.apply().add(new AMFExchangeDependencyResourceLoader(rootDir));

    if (format.equals(RAML)) {
      return new Raml10Parser(env);
    } else if (format.equals(JSON_OAS)) {
      return new Oas20Parser(env);
    } else if (format.equals(YAML_OAS)) {
      return new Oas20YamlParser(env);
    } else if (format.equals(JSON_OAS3)) {
      return new Oas30Parser(env);
    } else { // if (format.equals(YAML_OAS3)
      return new Oas30YamlParser(env);
    }
  }

  private WebApi getWebApi(Parser parser, Path path, boolean skipValidations) throws InvalidSourceException {
    if (Files.notExists(path)) {
      throw new InvalidSourceException(String.format("Spec file [%s] does not exist", path.toAbsolutePath()));
    }
    if (isRaml08(path.toAbsolutePath())) {
      throw new InvalidSourceException("RAML 0.8 is not supported.");
    }

    Document document;
    try {
      document = parseFile(parser, "file://" + path.toAbsolutePath().toString());
    } catch (Exception e) {
      throw new InvalidSourceException("Error parsing spec: " + e.getMessage() + ".", e);
    }
    if (!skipValidations) {
      validate(document, specFormat);
    }

    document = resolve(document, specFormat);
    return getWebApi(document);
  }

  private Document resolve(Document document, SpecFormat specFormat) {
    if (specFormat.equals(RAML)) {
      return (Document) new Raml10Resolver().resolve(document);
    } else if (specFormat.equals(JSON_OAS) || specFormat.equals(YAML_OAS)) {
      return (Document) new Oas20Resolver().resolve(document);
    } else { // if (specFormat.equals(JSON_OAS3) || specFormat.equals(YAML_OAS3)
      return (Document) new Oas30Resolver().resolve(document);
    }
  }

  private void validate(Document document, SpecFormat specFormat) throws InvalidSourceException {
    ProfileName profileName;
    switch (specFormat) {
      case RAML:
        profileName = ProfileNames.RAML();
        break;
      case JSON_OAS:
      case YAML_OAS:
        profileName = ProfileNames.OAS();
        break;
      case JSON_OAS3:
      case YAML_OAS3:
        profileName = ProfileNames.OAS30();
        break;
      default:
        throw new IllegalArgumentException(String.format("Spec format [%s] is unsupported", specFormat.getName()));
    }

    ValidationReport report;
    try {
      report = AMF.validate(document, profileName, ProfileNames.AMF().messageStyle()).get();
    } catch (ExecutionException | InterruptedException e) {
      throw new InvalidSourceException(String.format("Error validating the spec [%s], detail: %s.", document.location(),
                                                     e.getMessage()),
                                       e);
    }
    if (!report.conforms()) {
      String errorMessage = String.format("The API spec [%s] is invalid, please fix it (or ignore it).",
                                          document.location());
      LOGGER.debug(String.format("%s. Full report:\n%s", errorMessage, report.toString()));
      throw new InvalidSourceException(errorMessage);
    }
  }

  private WebApi getWebApi(Document document) {
    return (WebApi) document.encodes();
  }

  private Document parseFile(Parser parser, String url) {
    return handleFuture(parser.parseFileAsync(url));
  }

  private Document handleFuture(CompletableFuture<BaseUnit> f) {
    try {
      return (Document) f.get();
    } catch (ExecutionException e) {
      throw new RuntimeException("An error occurred while parsing the api. Message: " + e.getMessage(), e);
    } catch (InterruptedException e) {
      currentThread().interrupt();
      throw new RuntimeException("An error occurred while parsing the api. Message: " + e.getMessage(), e);
    }
  }

  private boolean isRaml08(Path specPath) throws InvalidSourceException {
    String content;

    try {
      content = new String(java.nio.file.Files.readAllBytes(specPath), StandardCharsets.UTF_8);
    } catch (IOException e) {
      throw new InvalidSourceException("Could not load raml file content", e);
    }

    return content.startsWith("#%RAML 0.8");
  }
}
