/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.model;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public abstract class APIOperationModel {

  protected String operationId;
  protected String name;
  protected String description;
  protected String summary;
  protected String path;
  protected String httpMethod;
  protected List<String> baseUris;

  protected List<APIParameterModel> uriParamsModel;
  protected List<APIParameterModel> queryParamsModel;
  protected List<APIParameterModel> headersModel;

  protected List<APITypeModel> inputMetadataModel;
  protected List<APITypeModel> outputMetadataModel;

  protected List<APISecuritySchemeModel> securitySchemesModel;

  protected APIDocumentationModel apiDocumentationModel;

  public String getOperationId() {
    return operationId;
  }

  public String getName() {
    return name;
  }

  public String getDescription() {
    return description;
  }

  public String getSummary() {
    return summary;
  }

  public String getPath() {
    return path;
  }

  public String getHttpMethod() {
    return httpMethod;
  }

  public List<String> getBaseUris() {
    return baseUris;
  }

  public List<APIParameterModel> getUriParamsModel() {
    return uriParamsModel;
  }

  public List<APIParameterModel> getQueryParamsModel() {
    return queryParamsModel;
  }

  public List<APIParameterModel> getHeadersModel() {
    return headersModel;
  }

  public List<APITypeModel> getInputMetadataModel() {
    return inputMetadataModel;
  }

  public List<APITypeModel> getOutputMetadataModel() {
    return outputMetadataModel;
  }

  public List<APISecuritySchemeModel> getSecuritySchemesModel() {
    return securitySchemesModel;
  }

  public APIDocumentationModel getAPIDocumentationModel() {
    return apiDocumentationModel;
  }

  protected <T> List<T> selectSecurityRequirements(List<T> operationSchemes, List<T> endPointSchemes, List<T> globalSchemes) {
    // If the operation defines security, we use it.
    if (!operationSchemes.isEmpty()) {
      return operationSchemes;
    }

    // If the operation does not defines security, we try with the endpoint.
    else if (!endPointSchemes.isEmpty()) {
      return endPointSchemes;
    }

    // If the endpoint does not defines security, we try with the global config.
    else if (!globalSchemes.isEmpty()) {
      return globalSchemes;
    }

    // If the method is not secured we return an empty list.
    else {
      return new ArrayList<>();
    }
  }

  /**
   * Retrieves a parameter by its name and type.
   * <p>
   * Different parameter types have different case sensitivity (see ComparisonUtil.externalNameParamsComparison()).
   *
   * @param apiParameterType the type of the parameter, e.g. header, query, uri (i.e. path), body, etc.
   * @param name the name to look for
   * @return the parameter
   */
  public Optional<APIParameterModel> getParameter(APIParameterType apiParameterType, String name) {
    List<APIParameterModel> parameterList;
    switch (apiParameterType) {
      case HEADER:
        parameterList = getHeadersModel();
        break;
      case QUERY:
        parameterList = getQueryParamsModel();
        break;
      case URI:
        parameterList = getUriParamsModel();
        break;
      default:
        throw new IllegalArgumentException();
    }
    return parameterList.stream().filter(p -> apiParameterType.areNamesEqual(p.getExternalName(), name)).findFirst();
  }

  /**
   * Retrieves the location in the API Spec this element was parsed from.
   */
  public abstract APILocation getLocation();
}
