/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIType.MULTIPART;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA_TYPE;
import static org.slf4j.LoggerFactory.getLogger;

import amf.core.client.platform.model.domain.Graph;
import amf.shapes.client.platform.model.domain.AnyShape;
import amf.shapes.client.platform.model.domain.ArrayShape;
import amf.shapes.client.platform.model.domain.FileShape;
import amf.shapes.client.platform.model.domain.NodeShape;
import amf.shapes.client.platform.model.domain.ScalarShape;
import amf.shapes.client.platform.model.domain.SchemaShape;
import amf.shapes.client.platform.model.domain.UnionShape;
import amf.shapes.client.platform.model.domain.Example;
import amf.core.client.platform.model.domain.DataNode;
import amf.core.client.platform.model.domain.Shape;
import amf.core.client.platform.model.domain.PropertyShape;
import amf.core.client.platform.model.domain.ScalarNode;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.MalformedSpecException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APILocation;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveTypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;
import org.slf4j.Logger;

import javax.ws.rs.core.MediaType;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

public class AMFTypeModel extends APITypeModel {

  private static final Logger LOGGER = getLogger(AMFTypeModel.class);

  private AnyShape shape;
  private boolean extractExample;

  public AMFTypeModel(APIPrimitiveTypeModel primitiveTypeModel) {
    super(primitiveTypeModel);
  }

  public AMFTypeModel(AnyShape anyShape, String mediaType, boolean extractExample) {
    this(anyShape, mediaType, null, extractExample);
  }

  public AMFTypeModel(AnyShape anyShape, String mediaType, Boolean required, boolean extractExample) {
    this.shape = anyShape;

    this.mediaType = getMediaTypeForStringOrNull(mediaType);
    this.extractExample = extractExample;
    if (extractExample) {
      this.example = buildResponseExample();
    }
    List<DataNode> values = shape.values();
    this.enumValues = !values.isEmpty() ? values.stream().map(AMFTypeModel::getEnumValue).collect(toList()) : null;
    this.APITypeSchemaModel = AMFTypeSchemaModel.builder().build(anyShape, this.mediaType);
    this.apiType = buildTypeDefinitionClass(this.mediaType);
    this.displayName = buildDisplayName();
    this.description = buildDescription();
    this.required = required;
  }

  private String buildDescription() {
    return shape.description().isNullOrEmpty() ? null : shape.description().value();
  }

  private String buildDisplayName() {
    return shape.displayName().isNullOrEmpty() ? null : shape.displayName().value();
  }

  private APIType buildTypeDefinitionClass(MediaType mediaType) {
    if (shape instanceof ScalarShape || shape instanceof FileShape) {
      this.primitiveTypeModel = new AMFPrimitiveTypeModel(shape);
      return APIType.PRIMITIVE;
    }

    else if (MULTIPART_FORM_DATA_TYPE.equals(mediaType)
        && shape instanceof NodeShape
        && !((NodeShape) shape).properties().isEmpty()) {
      return MULTIPART;
    }

    else if (shape instanceof NodeShape || shape instanceof SchemaShape) {
      return APIType.OBJECT_TYPE;
    }

    else if (shape instanceof UnionShape) {
      return APIType.UNION_TYPE;
    }

    else if (shape instanceof ArrayShape) {
      return APIType.ARRAY;
    }

    else if (!shape.and().isEmpty() || !shape.or().isEmpty() || !shape.xone().isEmpty()) {
      return APIType.OBJECT_TYPE;
    }

    // This is for the case that the type is not defined (for instance, when there is an example but not a definition).
    // Also, when the type is defined as 'nil'.
    else {
      return APIType.EMPTY;
    }
  }

  private String buildResponseExample() {
    return shape.examples().stream()
        .filter(x -> mediaType == null || (isYamlExample(x) || isRamlExample(x)))
        .findFirst().map(value -> value.value().value()).orElse(null);
  }

  private boolean isRamlExample(Example example) {
    return example.mediaType().value() == null;
  }

  private boolean isYamlExample(Example example) {
    return example.mediaType().value() != null && example.mediaType().value().equals(this.mediaType.toString());
  }

  @Override
  public List<APIParameterModel> getParts() {
    if (apiType.equals(MULTIPART)) {
      List<APIParameterModel> parts = new ArrayList<>();

      for (PropertyShape s : ((NodeShape) shape).properties()) {
        parts.add(new AMFParameterModel(s, APIParameterType.PART));
      }

      return parts;
    } else {
      return emptyList();
    }
  }

  @Override
  public AMFTypeModel getInnerType() {
    if (shape instanceof ArrayShape) {
      return fromShape(((ArrayShape) shape).items());
    } else {
      return null;
    }
  }

  @Override
  public List<APITypeModel> getUnionTypes() {
    if (shape instanceof UnionShape) {
      return ((UnionShape) shape).anyOf().stream().map(this::fromShape)
          .collect(collectingAndThen(toList(), Collections::unmodifiableList));
    } else {
      return emptyList();
    }
  }

  @Override
  public Map<String, APITypeModel> getObjectProperties() {
    if (!(shape instanceof NodeShape))
      return Collections.emptyMap();
    return (((NodeShape) shape)).properties().stream()
        .collect(collectingAndThen(toMap(ps -> ps.name().value(), ps -> fromShape(ps.range())), Collections::unmodifiableMap));
  }

  private AMFTypeModel fromShape(Shape innerShape) {
    if (innerShape instanceof AnyShape) {
      return new AMFTypeModel((AnyShape) innerShape, null, this.extractExample);
    } else {
      // Defaulting to string type for shapes that dont implement AnyShape.
      // These shapes can not generate a json/xml schema.
      // TODO: RSDK-8: Recursive shapes can't generate schemas.
      if (innerShape == null) {
        LOGGER.warn("Shape {} has no inner shape (e.g. an array without item type); treating it as string. At {}.", shape,
                    APILocation.from(shape.annotations()));
      } else {
        LOGGER.warn("Shape {} is not supported (within {}); treating it as string. At {}.", innerShape, shape,
                    APILocation.from(innerShape.annotations()));
      }
      return new AMFTypeModel(new AMFPrimitiveTypeModel(APIPrimitiveType.STRING));
    }
  }

  private static String getEnumValue(DataNode x) {
    if (x instanceof ScalarNode) {
      return ((ScalarNode) x).value().value();
    }
    throw new MalformedSpecException("Enum value type (" + x.getClass() + ") not supported", APILocation.from(x.annotations()));
  }

  public Graph graph() {
    return shape.graph();
  }

  @Override
  public APILocation getLocation() {
    return APILocation.from(shape.annotations());
  }
}
