/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.model;

import org.apache.commons.lang3.tuple.Pair;

import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

public abstract class APIModel {

  private final String apiName;
  private final String description;
  private final String baseUri;
  private final String apiVersion;
  private final List<APIOperationModel> operationsModel;
  private final List<String> protocols;

  private final Map<String, APIOperationModel> operationsByOperationId = new HashMap<>();
  private final Map<Pair<String, String>, APIOperationModel> operationsByMethodAndPath = new HashMap<>();

  public APIModel(String apiName, String description, String baseUri, String apiVersion, List<APIOperationModel> operations,
                  List<String> protocols) {
    this.apiName = apiName;
    this.description = description;
    this.baseUri = baseUri;
    this.apiVersion = apiVersion;
    this.operationsModel = operations;
    this.protocols = protocols;
    for (APIOperationModel apiOperationModel : operations) {
      APIOperationModel prev = operationsByMethodAndPath.put(
                                                             Pair.of(apiOperationModel.getHttpMethod().toLowerCase(Locale.ROOT),
                                                                     apiOperationModel.getPath().toLowerCase(Locale.ROOT)),
                                                             apiOperationModel);
      if (prev != null)
        throw new IllegalStateException("There are more than one operation with same path and http method, but different case. One of them has path '"
            + prev.getPath() + "' and method '" + prev.getHttpMethod() + "'");
      String operationId = apiOperationModel.getOperationId();
      if (operationId != null)
        operationsByOperationId.put(operationId, apiOperationModel);
    }
  }

  public String getApiName() {
    return apiName;
  }

  public String getDescription() {
    return description;
  }

  public List<String> getProtocols() {
    return protocols;
  }

  public String getBaseUri() {
    return baseUri;
  }

  public String getApiVersion() {
    return apiVersion;
  }

  public List<APIOperationModel> getOperationsModel() {
    return operationsModel;
  }

  /**
   * Retrieves the location in the API Spec this element was parsed from.
   */
  public abstract APILocation getLocation();

  /**
   * Finds an operation by operationId.
   *
   * @param operationId the operationId
   * @return an {@link APIOperationModel} with the given operationId
   */
  public Optional<APIOperationModel> findOperation(String operationId) {
    return Optional.ofNullable(operationsByOperationId.get(Objects.requireNonNull(operationId, "operationId must not be null")));
  }

  /**
   * Finds an operation by method and path.
   *
   * @param path an endpoint path
   * @param method an HTTP method
   * @return an {@link APIOperationModel} with the given method and path
   */
  public Optional<APIOperationModel> findOperation(String path, String method) {
    return Optional.ofNullable(operationsByMethodAndPath
        .get(Pair.of(Objects.requireNonNull(method, "method must not be null").toLowerCase(Locale.ROOT),
                     Objects.requireNonNull(path, "path must not be null").toLowerCase(Locale.ROOT))));
  }
}
