/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APILocation;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.schema.APITypeSchemaModel;

import java.util.List;
import java.util.Map;

import javax.ws.rs.core.MediaType;

import org.slf4j.Logger;

public abstract class APITypeModel {

  private static final Logger LOGGER = getLogger(APITypeModel.class);

  protected APIType apiType;
  protected MediaType mediaType;
  protected String example;
  protected List<String> enumValues;
  protected com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.schema.APITypeSchemaModel APITypeSchemaModel;
  protected APIPrimitiveType primitiveType;
  protected List<APIParameterModel> parts;
  protected APITypeModel innerType;
  protected List<APITypeModel> unionTypes;
  protected String displayName;
  protected String description;
  protected Boolean required;

  protected APITypeModel() {

  }

  protected APITypeModel(APIPrimitiveType primitiveType) {
    this.apiType = APIType.PRIMITIVE;
    this.primitiveType = primitiveType;
  }

  protected MediaType getMediaTypeForStringOrNull(String mediaType) {
    try {
      return isNotBlank(mediaType) ? MediaType.valueOf(mediaType) : null;
    } catch (Exception e) {
      LOGGER.error("Error getting media type", e);
      return null;
    }
  }

  public APIType getApiType() {
    return apiType;
  }

  public MediaType getMediaType() {
    return mediaType;
  }

  public String getExample() {
    return example;
  }

  public List<String> getEnumValues() {
    return enumValues;
  }

  public APITypeSchemaModel getAPITypeSchemaModel() {
    return APITypeSchemaModel;
  }

  public APIPrimitiveType getPrimitiveType() {
    return primitiveType;
  }

  public List<APIParameterModel> getParts() {
    return parts;
  }

  public APITypeModel getInnerType() {
    return innerType;
  }

  public List<APITypeModel> getUnionTypes() {
    return unionTypes;
  }

  public String getDisplayName() {
    return displayName;
  }

  public String getDescription() {
    return description;
  }

  public Boolean isRequired() {
    return required;
  }

  /**
   * Retrieves the types of the properties of an object.
   *
   * @return a map from property name to property type
   */
  public abstract Map<String, APITypeModel> getObjectProperties();

  /**
   * Retrieves the location in the API Spec this element was parsed from.
   */
  public abstract APILocation getLocation();

  @Override
  public String toString() {
    StringBuilder stringBuilder = new StringBuilder(getApiType().toString());
    switch (getApiType()) {
      case OBJECT_TYPE:
        stringBuilder.append(getObjectProperties());
        break;
      case ARRAY:
        stringBuilder.append("[").append(getInnerType()).append(']');
        break;
    }
    return stringBuilder.toString();
  }


  /**
   * Traverses an {@link APITypeModel} tree.
   * <p>
   * The {@link Visitor#visit(APITypeModel, String)} method is called once for each element in the tree with the type and a
   * dot-separated path. The root element is also visited (with a path of "").
   */
  public void accept(Visitor visitor) {
    accept(visitor, "");
  }

  private void accept(Visitor visitor, String path) {
    visitor.visit(this, path);
    switch (getApiType()) {
      case OBJECT_TYPE:
        getObjectProperties().forEach((propertyName, propertyType) -> propertyType
            .accept(visitor, path.isEmpty() ? propertyName : path + '.' + propertyName));
        break;
      case ARRAY:
        APITypeModel innerType = getInnerType();
        if (innerType != null)
          innerType.accept(visitor, path);
        break;
    }
  }

  /**
   * This "visitor" is used to receive information about a type tree.
   *
   * @see #accept(Visitor)
   */
  public static abstract class Visitor {

    /**
     * Gives information about an APITypeModel element in the type tree.
     *
     * @param type the type
     * @param path the dot-separated path relative to the type whose {@link #accept(Visitor)} method was called
     */
    public abstract void visit(APITypeModel type, String path);
  }
}
