/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.ParserUtils.removeHtmlTags;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.isXmlSchema;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.APPLICATION_OCTET_STREAM;
import static javax.ws.rs.core.MediaType.APPLICATION_XML;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import amf.core.client.platform.model.StrField;
import amf.core.client.platform.model.domain.Graph;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APILocation;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType;

import amf.apicontract.client.platform.model.domain.Parameter;
import amf.apicontract.client.platform.model.domain.Payload;
import amf.core.client.platform.model.domain.PropertyShape;
import amf.core.client.platform.model.domain.Shape;
import amf.shapes.client.platform.model.domain.AnyShape;
import amf.shapes.client.platform.model.domain.ArrayShape;
import amf.shapes.client.platform.model.domain.FileShape;
import amf.shapes.client.platform.model.domain.NodeShape;
import amf.shapes.client.platform.model.domain.SchemaShape;
import amf.shapes.client.platform.model.domain.UnionShape;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveType;

import java.util.Objects;

public class AMFParameterModel extends APIParameterModel {

  private final Parameter parameter;

  public AMFParameterModel(Parameter parameter, APIParameterType parameterType, boolean isPassword) {
    this.parameter = parameter;
    this.externalName = parameter.parameterName().value();
    this.displayName = buildDisplayName(parameter);
    this.parameterType = parameterType;
    this.defaultValue = buildDefaultValue(parameter);
    this.description = buildDescription(parameter);
    this.isPassword = isPassword;
    this.isRequired = parameter.required().value();
    this.summary = buildDisplayName(parameter);

    Shape parameterShape = getParameterShape(parameter);
    this.typeModel = new AMFTypeModel((AnyShape) parameterShape, getDefaultMediaType(parameterShape),
                                      parameter.required().value(), false);
  }

  private static String buildDisplayName(Parameter parameter) {
    StrField displayName = getParameterShape(parameter).displayName();
    return displayName.isNullOrEmpty() ? null : removeHtmlTags(displayName.value());
  }

  private static String buildDescription(Parameter parameter) {
    return parameter.description().value();
  }

  public AMFParameterModel(PropertyShape propertyShape, APIParameterType parameterType) {
    this.externalName = propertyShape.id().substring(propertyShape.id().lastIndexOf('/') + 1);
    this.displayName = propertyShape.displayName().nonEmpty() ? propertyShape.displayName().value()
        : (propertyShape.range().displayName().nonEmpty() ? propertyShape.range().displayName().value() : null);
    this.description = propertyShape.description().nonEmpty() ? propertyShape.description().value()
        : propertyShape.range().description().value();
    this.parameterType = parameterType;
    this.typeModel =
        new AMFTypeModel((AnyShape) propertyShape.range(), getDefaultMediaType(propertyShape.range()),
                         false, false);
    this.summary = this.displayName;
    this.parameter = null;
  }

  public AMFParameterModel(String name, APIParameterType parameterType, APIPrimitiveType amfPrimitiveTypeParser) {
    this.externalName = name;
    this.displayName = null;
    this.parameterType = parameterType;
    this.typeModel = new AMFTypeModel(amfPrimitiveTypeParser);
    this.parameter = null;
  }

  private static String buildDefaultValue(Parameter parameter) {
    StrField defaultValue = getParameterShape(parameter).defaultValueStr();
    return defaultValue.nonEmpty() ? defaultValue.value() : EMPTY;
  }

  private static String getDefaultMediaType(Shape shape) {
    if (shape instanceof FileShape)
      return APPLICATION_OCTET_STREAM;

    if (shape instanceof NodeShape || shape instanceof ArrayShape || shape instanceof UnionShape)
      return APPLICATION_JSON;

    if (shape instanceof SchemaShape) {
      if (isXmlSchemaShape(shape))
        return APPLICATION_XML;
      else
        return APPLICATION_JSON;
    }

    return TEXT_PLAIN;
  }

  private static boolean isXmlSchemaShape(Shape shape) {
    if (!(shape instanceof SchemaShape)) {
      return false;
    }
    SchemaShape schemaShape = (SchemaShape) shape;
    return schemaShape.raw().nonEmpty() && isXmlSchema(schemaShape.raw().value());
  }

  private static Shape getParameterShape(Parameter parameter) {
    Shape shape = parameter.schema();
    return shape != null ? shape
        : parameter.payloads().stream().map(Payload::schema).filter(Objects::nonNull).findFirst().orElse(null);
  }

  @SuppressWarnings("unused")
  public Graph graph() {
    return parameter.graph();
  }

  @Override
  public APILocation getLocation() {
    return APILocation.from(parameter.annotations());
  }
}
