/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType.URI;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.ParserUtils.removeHtmlTags;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.UnsupportedSecuritySchemeException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.security.AMFSecuritySchemesNaming;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Stream;

import amf.client.model.StrField;
import amf.client.model.domain.AnyShape;
import amf.client.model.domain.DomainElement;
import amf.client.model.domain.EndPoint;
import amf.client.model.domain.OAuth2Settings;
import amf.client.model.domain.Operation;
import amf.client.model.domain.Parameter;
import amf.client.model.domain.ParametrizedSecurityScheme;
import amf.client.model.domain.Payload;
import amf.client.model.domain.Response;
import amf.client.model.domain.SecurityRequirement;
import amf.client.model.domain.SecurityScheme;

public class AMFOperationModel extends APIOperationModel {

  private final EndPoint endPoint;
  private final Operation operation;

  public AMFOperationModel(EndPoint endPoint, Operation operation)
      throws ModelGenerationException {
    this.endPoint = endPoint;
    this.operation = operation;

    this.description = buildOperationDescription();
    this.summary = operation.summary().value();
    this.path = buildOperationPath();
    this.httpMethod = operation.method().value();
    this.name = buildName();
    this.operationId = operation.operationId().value();
    this.uriParamsModel = buildUriParamsModel();
    this.queryParamsModel = buildQueryParamsModel();
    this.headersModel = buildHeadersModel();
    this.inputMetadataModel = buildInputMetadataModels();
    this.outputMetadataModel = buildOutputMetadataModels();
    this.securitySchemesModel = buildSecuritySchemesModel();
  }

  private List<APISecuritySchemeModel> buildSecuritySchemesModel() throws ModelGenerationException {

    List<SecurityScheme> endPointSchemes =
        endPoint.security().stream().flatMap(AMFOperationModel::getSchemes).filter(Objects::nonNull).collect(toList());

    List<SecurityScheme> operationSchemes =
        operation.security().stream().flatMap(AMFOperationModel::getSchemes).filter(Objects::nonNull).collect(toList());

    List<SecurityScheme> selectedSchemes = selectSecuritySchemes(operationSchemes, endPointSchemes, new ArrayList<>());

    List<APISecuritySchemeModel> securitySchemesModel = new ArrayList<>();
    for (SecurityScheme securityScheme : selectedSchemes) {
      if (AMFSecuritySchemesNaming.isOauth2(securityScheme.type().value())) {
        securitySchemesModel.addAll(((OAuth2Settings) securityScheme.settings())
            .flows().stream()
            .map(x -> new AMFSecuritySchemeModel(securityScheme, x))
            .collect(toList()));
      } else {
        securitySchemesModel.add(new AMFSecuritySchemeModel(securityScheme));
      }
    }

    // If the operation is secured but we don't support any defined schemes, then we must throw a generation exception.
    if (!selectedSchemes.isEmpty() && securitySchemesModel.isEmpty()) {
      throw new UnsupportedSecuritySchemeException(
                                                   this.httpMethod + " " + this.path + ": " +
                                                       "None of the specified security schemes are supported.");
    }

    return securitySchemesModel;
  }

  private static Stream<SecurityScheme> getSchemes(DomainElement domainElement) {
    if (domainElement instanceof SecurityRequirement) {
      SecurityRequirement securityRequirement = (SecurityRequirement) domainElement;
      return securityRequirement.schemes().stream().map(ParametrizedSecurityScheme::scheme);
    }
    return Stream.empty();
  }

  private List<APITypeModel> buildOutputMetadataModels() {
    List<APITypeModel> models = new ArrayList<>();
    List<APITypeModel> defaultModel = new ArrayList<>();

    if (operation.responses() != null) {
      for (Response response : operation.responses()) {
        if (response.statusCode().nonEmpty()) {
          if (response.statusCode().value().startsWith("2")) {
            models.addAll(buildAMFTypeModel(response));
          } else if (response.statusCode().value().startsWith("default")) {
            defaultModel.addAll(buildAMFTypeModel(response));
          }
        }
      }
    }

    if (models.isEmpty()) {
      return defaultModel;
    }

    return models;
  }

  private List<APITypeModel> buildAMFTypeModel(Response response) {
    return buildIOMetadataModels(response.payloads(),
                                 operation.contentType());
  }

  private List<APITypeModel> buildInputMetadataModels() {
    if (operation.request() != null) {
      return buildIOMetadataModels(operation.request().payloads(),
                                   operation.accepts());
    }
    return emptyList();
  }

  private List<APITypeModel> buildIOMetadataModels(List<Payload> payloads, List<StrField> globalMediaTypes) {
    List<APITypeModel> models = new ArrayList<>();

    if (payloads == null) {
      return models;
    }

    List<String> globalMediaTypesString = globalMediaTypes.stream().map(StrField::value).collect(toList());

    for (Payload payload : payloads) {
      if (payload.schema() != null) {
        if (payload.mediaType().isNullOrEmpty() && !globalMediaTypesString.isEmpty()) {
          globalMediaTypesString.forEach(x -> models.add(new AMFTypeModel((AnyShape) payload.schema(), x)));
        } else {
          models.add(new AMFTypeModel((AnyShape) payload.schema(), payload.mediaType().value()));
        }
      }
    }

    return models;
  }

  private String buildName() {
    if (operation.name().nonEmpty() && isNotBlank(operation.name().value())
        && !operation.name().value().equalsIgnoreCase(operation.method().value())) {
      return operation.name().value();
    }
    return null;
  }

  private String buildOperationPath() {
    return endPoint.path().value();
  }

  private String buildOperationDescription() {
    final String description =
        operation.description().nonEmpty() ? operation.description().value() : endPoint.description().value();
    return removeHtmlTags(description);
  }

  private List<APIParameterModel> buildUriParamsModel() {
    List<APIParameterModel> list = new ArrayList<>();

    if (operation.request() != null) {
      for (Parameter x : operation.request().uriParameters()) {
        AMFParameterModel parameterModel = new AMFParameterModel(x, URI, false);
        list.add(parameterModel);
      }
    }

    for (Parameter x : endPoint.parameters()) {
      AMFParameterModel parameterModel = new AMFParameterModel(x, URI, false);
      list.add(parameterModel);
    }
    return list;
  }

  private List<APIParameterModel> buildQueryParamsModel() {
    List<APIParameterModel> list = new ArrayList<>();
    if (operation.request() != null) {
      for (Parameter x : operation.request().queryParameters()) {
        AMFParameterModel parameterModel = new AMFParameterModel(x, QUERY, false);
        list.add(parameterModel);
      }
    }
    return list;
  }

  private List<APIParameterModel> buildHeadersModel() {
    List<APIParameterModel> list = new ArrayList<>();
    if (operation.request() != null) {
      for (Parameter x : operation.request().headers()) {
        AMFParameterModel parameterModel = new AMFParameterModel(x, HEADER, false);
        list.add(parameterModel);
      }
    }
    return list;
  }
}
