/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.BASIC_AUTH;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.BEARER_AUTH;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.CUSTOM;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.DIGEST;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.NOT_SUPPORTED;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.OAUTH2_AUTHORIZATION_CODE;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.OAUTH2_CLIENT_CREDENTIALS;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType.PASS_THROUGH;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType.SECURITY;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveType.STRING;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.security.AMFOauth2FlowsNaming;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.security.AMFSecuritySchemesNaming;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;

import amf.apicontract.client.platform.model.domain.security.ApiKeySettings;
import amf.apicontract.client.platform.model.domain.security.HttpSettings;
import amf.apicontract.client.platform.model.domain.security.OAuth2Flow;
import amf.apicontract.client.platform.model.domain.security.OAuth2Settings;
import amf.apicontract.client.platform.model.domain.security.SecurityScheme;
import amf.apicontract.client.platform.model.domain.Parameter;

public class AMFSecuritySchemeModel extends APISecuritySchemeModel {

  private final OAuth2Flow oAuth2Flow;

  public AMFSecuritySchemeModel(SecurityScheme securityScheme) {
    this(securityScheme, null);
  }

  public AMFSecuritySchemeModel(List<SecurityScheme> securitySchemas) {
    this.oAuth2Flow = null;
    initModel();
    for (SecurityScheme securityScheme : securitySchemas) {
      loadModel(securityScheme);
    }
  }

  public AMFSecuritySchemeModel(SecurityScheme securityScheme, OAuth2Flow oAuth2Flow) {
    this.oAuth2Flow = oAuth2Flow;
    initModel();
    loadModel(securityScheme);
  }

  private void initModel() {
    this.customHeaders = new ArrayList<>();
    this.customQueryParams = new ArrayList<>();
  }

  private void loadModel(SecurityScheme securityScheme) {
    this.name = buildName(securityScheme);
    this.securitySchemeType = buildSecuritySchemeClass(securityScheme);
    this.customHeaders.addAll(buildCustomHeadersModel(securityScheme));
    this.customQueryParams.addAll(buildCustomQueryParamsModel(securityScheme));
    this.accessTokenUri = buildAccessTokenUri();
    this.authorizationUri = buildAuthorizationUri();
    this.scopes = buildScopes();
  }

  private String buildName(SecurityScheme securityScheme) {
    String partialName = securityScheme.name().value();
    if (StringUtils.isEmpty(name)) {
      return partialName;
    } else {
      return this.name.concat("-" + partialName);
    }
  }

  private String buildAccessTokenUri() {
    if (oAuth2Flow != null) {
      return oAuth2Flow.accessTokenUri().value();
    }
    return null;
  }

  private String buildAuthorizationUri() {
    if (oAuth2Flow != null) {
      return oAuth2Flow.authorizationUri().value();
    }
    return null;
  }

  private List<String> buildScopes() {
    if (oAuth2Flow != null) {
      return oAuth2Flow.scopes().stream().map(x -> x.name().value()).collect(toList());
    }
    return emptyList();
  }

  private APISecuritySchemeType buildSecuritySchemeClass(SecurityScheme securityScheme) {
    String schemeType = securityScheme.type().value();
    if (AMFSecuritySchemesNaming.isBasicAuth(schemeType, getHttpSettingsScheme(securityScheme))) {
      return BASIC_AUTH;
    } else if (AMFSecuritySchemesNaming.isPassThrough(schemeType)) {
      return PASS_THROUGH;
    } else if (AMFSecuritySchemesNaming.isApiKey(schemeType)) {
      return PASS_THROUGH;
    } else if (AMFSecuritySchemesNaming.isOauth2(schemeType)) {
      OAuth2Settings oAuth2Settings = (OAuth2Settings) securityScheme.settings();
      if (AMFOauth2FlowsNaming.isAuthorizationCode(oAuth2Settings, oAuth2Flow)) {
        return OAUTH2_AUTHORIZATION_CODE;
      } else if (AMFOauth2FlowsNaming.isClientCredentials(oAuth2Settings, oAuth2Flow)) {
        return OAUTH2_CLIENT_CREDENTIALS;
      }
    } else if (AMFSecuritySchemesNaming.isDigest(schemeType)) {
      return DIGEST;
    } else if (AMFSecuritySchemesNaming.isBearer(schemeType, getHttpSettingsScheme(securityScheme))) {
      return BEARER_AUTH;
    } else if (AMFSecuritySchemesNaming.isCustom(schemeType)) {
      return CUSTOM;
    }
    return NOT_SUPPORTED;
  }

  private String getHttpSettingsScheme(SecurityScheme securityScheme) {
    String httpScheme = null;
    if (securityScheme.settings() instanceof HttpSettings) {
      httpScheme = ((HttpSettings) securityScheme.settings()).scheme().value();
    }
    return httpScheme;
  }

  private List<APIParameterModel> buildCustomQueryParamsModel(SecurityScheme securityScheme) {
    List<APIParameterModel> list = new ArrayList<>();
    for (Parameter x : securityScheme.queryParameters()) {
      AMFParameterModel parameterModel = new AMFParameterModel(x, QUERY, false);
      list.add(parameterModel);
    }

    AMFParameterModel queryParam = getApiKeyParameter("query", securityScheme);
    if (queryParam != null) {
      list.add(queryParam);
    }

    return list;
  }

  private List<APIParameterModel> buildCustomHeadersModel(SecurityScheme securityScheme) {
    List<APIParameterModel> list = new ArrayList<>();
    for (Parameter x : securityScheme.headers()) {
      AMFParameterModel parameterModel = new AMFParameterModel(x, HEADER, false);
      list.add(parameterModel);
    }

    AMFParameterModel header = getApiKeyParameter("header", securityScheme);
    if (header != null) {
      list.add(header);
    }

    return list;
  }

  private AMFParameterModel getApiKeyParameter(String from, SecurityScheme securityScheme) {
    if (securityScheme.settings() instanceof ApiKeySettings) {
      String in = ((ApiKeySettings) securityScheme.settings()).in().value();
      String name = ((ApiKeySettings) securityScheme.settings()).name().value();
      name = isNotBlank(name) ? name : "api-key";

      if (in.equalsIgnoreCase(from)) {
        return new AMFParameterModel(name, SECURITY, new AMFPrimitiveTypeModel(STRING));
      }
    }

    return null;
  }
}
