/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static amf.apicontract.client.platform.WebAPIConfiguration.WebAPI;
import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.nio.file.Files.notExists;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.InvalidSourceException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.resourceloader.AMFExchangeDependencyResourceLoader;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;

import amf.apicontract.client.platform.WebAPIConfiguration;
import amf.apicontract.client.platform.model.domain.EndPoint;
import amf.apicontract.client.platform.model.domain.Server;
import amf.apicontract.client.platform.model.domain.api.WebApi;
import amf.apicontract.client.platform.AMFBaseUnitClient;
import amf.apicontract.client.platform.AMFConfiguration;
import amf.apicontract.client.scala.UnrecognizedSpecException;
import amf.core.client.platform.AMFParseResult;
import amf.core.client.platform.model.StrField;
import amf.core.client.platform.model.document.BaseUnit;
import amf.core.client.platform.model.document.Document;
import amf.core.client.platform.resource.FileResourceLoader;
import amf.core.client.platform.validation.AMFValidationReport;
import amf.core.client.platform.validation.AMFValidationResult;
import org.slf4j.Logger;

public class AMFAPIModel extends APIModel {

  public static final String AMF_VIOLATION_LEVEL = "VIOLATION";
  private static final Logger LOGGER = getLogger(AMFAPIModel.class);

  private static final AMFConfiguration AMF_WEB_API_CONFIG = WebAPI();

  private final WebApi webApi;

  public AMFAPIModel(File apiSpec, String rootDir, boolean skipValidations)
      throws ModelGenerationException {

    webApi = getWebApi(apiSpec.toPath(), rootDir, skipValidations);

    this.apiName = buildApiName();
    this.description = buildDescription();
    this.protocols = buildProtocols();
    this.apiVersion = buildApiVersion();
    this.baseUri = buildBaseUri();
    this.operationsModel = buildOperationsModel();
  }

  private String buildApiName() {
    return webApi.name().value();
  }

  private String buildDescription() {
    return webApi.description().nonEmpty() ? webApi.description().value() : EMPTY;
  }

  private String buildApiVersion() {
    return webApi.version() != null ? webApi.version().value() : null;
  }

  private String buildBaseUri() {
    String baseUri = null;

    Server server = webApi.servers().stream().filter(x -> x.url().nonEmpty()).findFirst().orElse(null);
    if (server != null) {
      baseUri = server.url().value();
    }

    return baseUri;
  }

  private List<String> buildProtocols() {
    return webApi.schemes().stream().map(StrField::value).collect(toList());
  }

  private List<APIOperationModel> buildOperationsModel() throws ModelGenerationException {
    List<APIOperationModel> operationsModel = new ArrayList<>();

    for (EndPoint endPoint : webApi.endPoints()) {
      for (amf.apicontract.client.platform.model.domain.Operation operation : endPoint.operations()) {
        operationsModel.add(new AMFOperationModel(endPoint, operation));
      }
    }

    return operationsModel;
  }

  private static void configureResourceLoaders(AMFConfiguration amfConfiguration, String rootDir) {
    amfConfiguration.withResourceLoaders(asList(new AMFExchangeDependencyResourceLoader(rootDir), new FileResourceLoader()));
  }

  public WebApi getWebApi(Path path, String rootDir, boolean skipValidations) throws InvalidSourceException {
    if (notExists(path)) {
      throw new InvalidSourceException(format("Spec file [%s] does not exist", path.toAbsolutePath()));
    }

    if (isRaml08(path.toAbsolutePath())) {
      throw new InvalidSourceException("RAML 0.8 is not supported.");
    }

    return parseWebApi(path, rootDir, skipValidations);
  }

  private WebApi parseWebApi(Path path, String rootDir, boolean skipValidations) throws InvalidSourceException {
    AMFParseResult amfParseResult;
    AMFConfiguration specificConfiguration;

    try {
      configureResourceLoaders(AMF_WEB_API_CONFIG, rootDir);
      amfParseResult = AMF_WEB_API_CONFIG.baseUnitClient().parse("file://" + path.toAbsolutePath()).get();
      specificConfiguration = WebAPIConfiguration.fromSpec(amfParseResult.sourceSpec());
    } catch (InterruptedException | ExecutionException e) {
      throw new InvalidSourceException("Error parsing spec: " + e.getMessage(), e);
    } catch (UnrecognizedSpecException e) {
      throw new InvalidSourceException("Error parsing spec: " + e.getMessage() + ". Supported encoding: UTF-8.", e);
    }

    if (!amfParseResult.conforms()) {
      throw new InvalidSourceException("The provided API specification has errors.\n" +
          amfParseResult.results().stream()
              .map(AMFValidationResult::message)
              .collect(joining(lineSeparator())));
    }

    AMFBaseUnitClient specBaseUnitClient = specificConfiguration.baseUnitClient();
    BaseUnit baseUnit = specBaseUnitClient.transform(amfParseResult.baseUnit()).baseUnit();

    Document document;
    if (baseUnit instanceof Document) {
      document = (Document) baseUnit;
    } else {
      throw new InvalidSourceException("Document type not supported: " + baseUnit.getClass());
    }

    if (!skipValidations) {
      validateDocument(specBaseUnitClient, document);
    }

    return (WebApi) document.encodes();
  }

  private void validateDocument(AMFBaseUnitClient client, Document document) throws InvalidSourceException {
    AMFValidationReport report;

    try {
      report = client.validate(document).get();
    } catch (Exception e) {
      String message = format("Error validating the spec [%s], detail: %s.", document.location(), e.getMessage());
      throw new InvalidSourceException(message);
    }

    if (!report.conforms()) {
      printErrors(report);

      String errorMessage = format("The API spec [%s] is invalid, please fix it (or ignore it).", document.location());
      throw new InvalidSourceException(errorMessage);
    }
  }

  private static void printErrors(AMFValidationReport report) {
    for (AMFValidationResult result : report.results()) {
      if (result.severityLevel().equalsIgnoreCase(AMF_VIOLATION_LEVEL)) {
        LOGGER.error(report.toString());
      }
    }
  }

  private boolean isRaml08(Path specPath) throws InvalidSourceException {
    String content;

    try {
      content = new String(java.nio.file.Files.readAllBytes(specPath), StandardCharsets.UTF_8);
    } catch (IOException e) {
      throw new InvalidSourceException("Could not load raml file content", e);
    }

    return content.startsWith("#%RAML 0.8");
  }
}
