/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType.URI;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.UnsupportedSecuritySchemeException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APISecuritySchemeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.security.AMFSecuritySchemesNaming;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import amf.core.client.platform.model.StrField;
import amf.shapes.client.platform.model.domain.AnyShape;
import amf.core.client.platform.model.domain.DomainElement;
import amf.apicontract.client.platform.model.domain.EndPoint;
import amf.apicontract.client.platform.model.domain.Operation;
import amf.apicontract.client.platform.model.domain.Parameter;
import amf.apicontract.client.platform.model.domain.Payload;
import amf.apicontract.client.platform.model.domain.Response;
import amf.apicontract.client.platform.model.domain.security.SecurityRequirement;
import amf.apicontract.client.platform.model.domain.security.SecurityScheme;
import amf.apicontract.client.platform.model.domain.security.OAuth2Settings;
import amf.apicontract.client.platform.model.domain.security.ParametrizedSecurityScheme;

public class AMFOperationModel extends APIOperationModel {

  private final EndPoint endPoint;
  private final Operation operation;

  public AMFOperationModel(EndPoint endPoint, Operation operation)
      throws ModelGenerationException {
    this.endPoint = endPoint;
    this.operation = operation;

    this.description = buildOperationDescription();
    this.summary = operation.summary().value();
    this.path = buildOperationPath();
    this.httpMethod = operation.method().value();
    this.baseUris = operation.servers().stream()
        .filter(s -> s.url() != null)
        .map(s -> s.url().value())
        .collect(Collectors.toList());
    this.name = buildName();
    this.operationId = operation.operationId().value();
    this.uriParamsModel = buildUriParamsModel();
    this.queryParamsModel = buildQueryParamsModel();
    this.headersModel = buildHeadersModel();
    this.inputMetadataModel = buildInputMetadataModels();
    this.outputMetadataModel = buildOutputMetadataModels();
    this.securitySchemesModel = buildSecuritySchemesModel();
  }


  private List<APISecuritySchemeModel> buildSecuritySchemesModel() throws ModelGenerationException {

    // Get from AMF web api endpoints and operations security requirements
    List<SecurityRequirement> endPointSecurityRequirements = endPoint.security();
    List<SecurityRequirement> operationSecurityRequirements = operation.security();

    // Filtering requirements between operations and endpoints and with not null schemes
    List<SecurityRequirement> selectedRequirements =
        selectSecurityRequirements(
                                   operationSecurityRequirements,
                                   endPointSecurityRequirements,
                                   new ArrayList<>())
                                       .stream().filter(x -> !x.schemes().isEmpty()).collect(toList());

    // Splitting between multiples schemes and once scheme
    List<SecurityRequirement> selectedRequirementsMultipleSchemes = selectedRequirements
        .stream()
        .filter(x -> x.schemes().size() > 1 && notNullSchemes(x.schemes()))
        .collect(toList());
    List<SecurityRequirement> selectedRequirementsSimpleSchemes = selectedRequirements
        .stream()
        .filter(x -> x.schemes().size() == 1 && notNullSchemes(x.schemes()))
        .collect(toList());

    List<APISecuritySchemeModel> securitySchemesModel = new ArrayList<>();

    for (SecurityRequirement securityRequirement : selectedRequirementsMultipleSchemes) {
      // Validate all schemas are apikey, other cases, not supported
      if (securityRequirement.schemes().stream().allMatch(x -> AMFSecuritySchemesNaming.isApiKey(x.scheme().type().value()))) {
        securitySchemesModel
            .add(new AMFSecuritySchemeModel(securityRequirement.schemes().stream().map(x -> x.scheme()).collect(toList())));
      }
    }
    for (SecurityRequirement securityRequirement : selectedRequirementsSimpleSchemes) {
      SecurityScheme securityScheme = securityRequirement.schemes().stream().findFirst().get().scheme();
      if (securityScheme != null) {
        if (AMFSecuritySchemesNaming.isOauth2(securityScheme.type().value())) {
          securitySchemesModel.addAll(((OAuth2Settings) securityScheme.settings())
              .flows().stream()
              .map(x -> new AMFSecuritySchemeModel(securityScheme, x))
              .collect(toList()));
        } else {
          securitySchemesModel.add(new AMFSecuritySchemeModel(securityScheme));
        }
      }
    }

    // If the operation is secured but we don't support any defined schemes, then we must throw a generation exception.
    if ((!selectedRequirementsMultipleSchemes.isEmpty() || !selectedRequirementsSimpleSchemes.isEmpty())
        && securitySchemesModel.isEmpty()) {
      throw new UnsupportedSecuritySchemeException(
                                                   this.httpMethod + " " + this.path + ": " +
                                                       "None of the specified security schemes are supported.");
    }

    return securitySchemesModel;
  }

  private boolean notNullSchemes(List<ParametrizedSecurityScheme> schemes) {
    return schemes.stream().anyMatch(y -> y.scheme() != null);
  }

  private static Stream<SecurityScheme> getSchemes(DomainElement domainElement) {
    if (domainElement instanceof SecurityRequirement) {
      SecurityRequirement securityRequirement = (SecurityRequirement) domainElement;
      return securityRequirement.schemes().stream().map(ParametrizedSecurityScheme::scheme);
    }
    return Stream.empty();
  }

  private List<APITypeModel> buildOutputMetadataModels() {
    List<APITypeModel> models = new ArrayList<>();
    List<APITypeModel> defaultModel = new ArrayList<>();

    if (operation.responses() != null) {
      for (Response response : operation.responses()) {
        if (response.statusCode().nonEmpty()) {
          if (response.statusCode().value().startsWith("2")) {
            models.addAll(buildAMFTypeModel(response));
          } else if (response.statusCode().value().startsWith("default")) {
            defaultModel.addAll(buildAMFTypeModel(response));
          }
        }
      }
    }

    if (models.isEmpty()) {
      return defaultModel;
    }

    return models;
  }

  private List<APITypeModel> buildAMFTypeModel(Response response) {
    return buildIOMetadataModels(response.payloads(),
                                 operation.contentType());
  }

  private List<APITypeModel> buildInputMetadataModels() {
    if (operation.request() != null) {
      return buildIOMetadataModels(operation.request().payloads(),
                                   operation.accepts());
    }
    return emptyList();
  }

  private List<APITypeModel> buildIOMetadataModels(List<Payload> payloads, List<StrField> globalMediaTypes) {
    List<APITypeModel> models = new ArrayList<>();

    if (payloads == null) {
      return models;
    }

    List<String> globalMediaTypesString = globalMediaTypes.stream().map(StrField::value).collect(toList());

    for (Payload payload : payloads) {
      if (payload.schema() != null) {
        if (payload.mediaType().isNullOrEmpty() && !globalMediaTypesString.isEmpty()) {
          globalMediaTypesString.forEach(x -> models.add(new AMFTypeModel((AnyShape) payload.schema(), x)));
        } else {
          models.add(new AMFTypeModel((AnyShape) payload.schema(), payload.mediaType().value()));
        }
      }
    }

    return models;
  }

  private String buildName() {
    if (operation.name().nonEmpty() && isNotBlank(operation.name().value())
        && !operation.name().value().equalsIgnoreCase(operation.method().value())
        && !operation.name().value().equalsIgnoreCase(operation.operationId().value())) {
      return operation.name().value();
    }
    return null;
  }

  private String buildOperationPath() {
    return endPoint.path().value();
  }

  private String buildOperationDescription() {
    return operation.description().nonEmpty() ? operation.description().value() : endPoint.description().value();
  }

  private List<APIParameterModel> buildUriParamsModel() {
    List<APIParameterModel> list = new ArrayList<>();

    if (operation.request() != null) {
      for (Parameter x : operation.request().uriParameters()) {
        AMFParameterModel parameterModel = new AMFParameterModel(x, URI, false);
        list.add(parameterModel);
      }
    }

    for (Parameter x : endPoint.parameters()) {
      AMFParameterModel parameterModel = new AMFParameterModel(x, URI, false);
      list.add(parameterModel);
    }
    return list;
  }

  private List<APIParameterModel> buildQueryParamsModel() {
    List<APIParameterModel> list = new ArrayList<>();
    if (operation.request() != null) {
      for (Parameter x : operation.request().queryParameters()) {
        AMFParameterModel parameterModel = new AMFParameterModel(x, QUERY, false);
        list.add(parameterModel);
      }
    }
    return list;
  }

  private List<APIParameterModel> buildHeadersModel() {
    List<APIParameterModel> list = new ArrayList<>();
    if (operation.request() != null) {
      for (Parameter x : operation.request().headers()) {
        AMFParameterModel parameterModel = new AMFParameterModel(x, HEADER, false);
        list.add(parameterModel);
      }
    }
    return list;
  }

}
