/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat.JSON_OAS;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat.JSON_OAS3;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat.RAML;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat.YAML_OAS;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat.YAML_OAS3;
import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.nio.file.Files.notExists;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.slf4j.LoggerFactory.getLogger;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.SpecFormat;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.InvalidSourceException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf.resourceloader.AMFExchangeDependencyResourceLoader;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;


import amf.apicontract.client.platform.model.domain.EndPoint;
import amf.apicontract.client.platform.model.domain.Server;
import amf.apicontract.client.platform.model.domain.api.WebApi;
import amf.apicontract.client.platform.AMFBaseUnitClient;
import amf.apicontract.client.platform.AMFConfiguration;
import amf.apicontract.client.platform.OASConfiguration;
import amf.apicontract.client.platform.RAMLConfiguration;
import amf.core.client.platform.AMFParseResult;
import amf.core.client.platform.model.StrField;
import amf.core.client.platform.model.document.BaseUnit;
import amf.core.client.platform.model.document.Document;
import amf.core.client.platform.resource.FileResourceLoader;
import amf.core.client.platform.validation.AMFValidationReport;
import amf.core.client.platform.validation.AMFValidationResult;
import org.slf4j.Logger;

public class AMFAPIModel extends APIModel {

  public static final String AMF_VIOLATION_LEVEL = "VIOLATION";
  private static final Logger LOGGER = getLogger(AMFAPIModel.class);

  private final SpecFormat specFormat;
  private final WebApi webApi;

  public AMFAPIModel(File apiSpec, String rootDir, SpecFormat specFormat, boolean skipValidations)
      throws ModelGenerationException {
    this.specFormat = specFormat;

    AMFConfiguration configuration = getAMFConfiguration(specFormat, rootDir);
    webApi = getWebApi(configuration.baseUnitClient(), apiSpec.toPath(), skipValidations);

    this.apiName = buildApiName();
    this.description = buildDescription();
    this.protocols = buildProtocols();
    this.apiVersion = buildApiVersion();
    this.baseUri = buildBaseUri();
    this.operationsModel = buildOperationsModel();
  }

  private String buildApiName() {
    return webApi.name().value();
  }

  private String buildDescription() {
    return webApi.description().nonEmpty() ? webApi.description().value() : EMPTY;
  }

  private String buildApiVersion() {
    return webApi.version() != null ? webApi.version().value() : null;
  }

  private String buildBaseUri() {
    String baseUri = null;

    Server server = webApi.servers().stream().filter(x -> x.url().nonEmpty()).findFirst().orElse(null);
    if (server != null) {
      baseUri = server.url().value();
    }

    return baseUri;
  }

  private List<String> buildProtocols() {
    return webApi.schemes().stream().map(StrField::value).collect(toList());
  }

  private List<APIOperationModel> buildOperationsModel() throws ModelGenerationException {
    List<APIOperationModel> operationsModel = new ArrayList<>();

    for (EndPoint endPoint : webApi.endPoints()) {
      for (amf.apicontract.client.platform.model.domain.Operation operation : endPoint.operations()) {
        operationsModel.add(new AMFOperationModel(endPoint, operation));
      }
    }

    return operationsModel;
  }

  private static AMFConfiguration getAMFConfiguration(SpecFormat format, String rootDir) {
    AMFConfiguration amfConfiguration;

    if (format.equals(RAML)) {
      amfConfiguration = RAMLConfiguration.RAML10();
    } else if (format.equals(JSON_OAS) || format.equals(YAML_OAS)) {
      amfConfiguration = OASConfiguration.OAS20();
    } else if (format.equals(YAML_OAS3) || format.equals(JSON_OAS3)) {
      amfConfiguration = OASConfiguration.OAS30();
    } else {
      throw new IllegalArgumentException("Spec format " + format.getName() + " not supported");
    }

    return amfConfiguration
        .withResourceLoaders(asList(new AMFExchangeDependencyResourceLoader(rootDir), new FileResourceLoader()));
  }

  public WebApi getWebApi(AMFBaseUnitClient client, Path path, boolean skipValidations) throws InvalidSourceException {
    if (notExists(path)) {
      throw new InvalidSourceException(format("Spec file [%s] does not exist", path.toAbsolutePath()));
    }

    if (isRaml08(path.toAbsolutePath())) {
      throw new InvalidSourceException("RAML 0.8 is not supported.");
    }

    Document document = getDocument(client, path);

    if (!skipValidations) {
      validateDocument(client, document);
    }

    return getWebApi(document);
  }

  private Document getDocument(AMFBaseUnitClient client, Path path) throws InvalidSourceException {
    AMFParseResult amfParseResult;

    try {
      amfParseResult = client.parse("file://" + path.toAbsolutePath()).get();
    } catch (InterruptedException | ExecutionException e) {
      throw new InvalidSourceException("Error parsing spec: " + e.getMessage(), e);
    }

    if (!amfParseResult.conforms()) {
      throw new InvalidSourceException("The provided API specification has errors.\n" +
          amfParseResult.results().stream()
              .map(AMFValidationResult::message)
              .collect(joining(lineSeparator())));
    }

    BaseUnit baseUnit = client.transform(amfParseResult.baseUnit()).baseUnit();

    if (baseUnit instanceof Document) {
      return (Document) baseUnit;
    } else {
      throw new InvalidSourceException("Document type not supported: " + baseUnit.getClass());
    }
  }

  private void validateDocument(AMFBaseUnitClient client, Document document) throws InvalidSourceException {
    AMFValidationReport report;

    try {
      report = client.validate(document).get();
    } catch (Exception e) {
      String message = format("Error validating the spec [%s], detail: %s.", document.location(), e.getMessage());
      throw new InvalidSourceException(message);
    }

    if (!report.conforms()) {
      printErrors(report);

      String errorMessage = format("The API spec [%s] is invalid, please fix it (or ignore it).", document.location());
      throw new InvalidSourceException(errorMessage);
    }
  }

  private static void printErrors(AMFValidationReport report) {
    for (AMFValidationResult result : report.results()) {
      if (result.severityLevel().equalsIgnoreCase(AMF_VIOLATION_LEVEL)) {
        LOGGER.error(report.toString());
      }
    }
  }

  private WebApi getWebApi(Document document) {
    return (WebApi) document.encodes();
  }

  private boolean isRaml08(Path specPath) throws InvalidSourceException {
    String content;

    try {
      content = new String(java.nio.file.Files.readAllBytes(specPath), StandardCharsets.UTF_8);
    } catch (IOException e) {
      throw new InvalidSourceException("Could not load raml file content", e);
    }

    return content.startsWith("#%RAML 0.8");
  }
}
