/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.ParserUtils.removeHtmlTags;
import static com.mulesoft.connectivity.rest.sdk.internal.webapi.util.XmlUtils.isXmlSchema;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.APPLICATION_OCTET_STREAM;
import static javax.ws.rs.core.MediaType.APPLICATION_XML;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN;
import static org.apache.commons.lang3.StringUtils.EMPTY;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType;

import amf.apicontract.client.platform.model.domain.Parameter;
import amf.apicontract.client.platform.model.domain.Payload;
import amf.core.client.platform.model.domain.PropertyShape;
import amf.core.client.platform.model.domain.Shape;
import amf.shapes.client.platform.model.domain.AnyShape;
import amf.shapes.client.platform.model.domain.ArrayShape;
import amf.shapes.client.platform.model.domain.FileShape;
import amf.shapes.client.platform.model.domain.NodeShape;
import amf.shapes.client.platform.model.domain.SchemaShape;
import amf.shapes.client.platform.model.domain.UnionShape;

public class AMFParameterModel extends APIParameterModel {

  public AMFParameterModel(Parameter parameter, APIParameterType parameterType, boolean isPassword) {
    this.externalName = parameter.parameterName().value();
    this.displayName = buildDisplayName(parameter);
    this.parameterType = parameterType;
    this.defaultValue = buildDefaultValue(parameter);
    this.description = buildDescription(parameter);
    this.isPassword = isPassword;
    this.isRequired = parameter.required().value();
    this.summary = buildDisplayName(parameter);

    Shape parameterShape = getParameterShape(parameter);
    this.typeModel = new AMFTypeModel((AnyShape) parameterShape, getDefaultMediaType(parameterShape), false);
  }

  private String buildDisplayName(Parameter parameter) {
    Shape parameterShape = getParameterShape(parameter);

    String displayName =
        parameterShape.displayName().isNullOrEmpty() ? null : removeHtmlTags(parameterShape.displayName().value());

    return displayName;
  }

  private String buildDescription(Parameter parameter) {
    return parameter.description().value();
  }

  public AMFParameterModel(PropertyShape propertyShape, APIParameterType parameterType) {
    this.externalName = propertyShape.id().substring(propertyShape.id().lastIndexOf('/') + 1);
    this.displayName = propertyShape.displayName().nonEmpty() ? propertyShape.displayName().value()
        : (propertyShape.range().displayName().nonEmpty() ? propertyShape.range().displayName().value() : null);
    this.description = propertyShape.description().nonEmpty() ? propertyShape.description().value()
        : propertyShape.range().description().value();
    this.parameterType = parameterType;
    this.typeModel =
        new AMFTypeModel((AnyShape) propertyShape.range(), getDefaultMediaType(propertyShape.range()), false);
    this.summary = this.displayName;
  }

  public AMFParameterModel(String name, APIParameterType parameterType, AMFPrimitiveTypeModel amfPrimitiveTypeParser) {
    this.externalName = name;
    this.displayName = null;
    this.parameterType = parameterType;
    this.typeModel = new AMFTypeModel(amfPrimitiveTypeParser);
  }

  private String buildDefaultValue(Parameter parameter) {
    Shape parameterShape = getParameterShape(parameter);
    return parameterShape.defaultValueStr().nonEmpty() ? parameterShape.defaultValueStr().value() : EMPTY;
  }

  private String getDefaultMediaType(Shape shape) {
    if (shape instanceof FileShape)
      return APPLICATION_OCTET_STREAM;

    if (shape instanceof NodeShape || shape instanceof ArrayShape || shape instanceof UnionShape)
      return APPLICATION_JSON;

    if (shape instanceof SchemaShape) {
      if (isXmlSchemaShape(shape))
        return APPLICATION_XML;
      else
        return APPLICATION_JSON;
    }

    return TEXT_PLAIN;
  }

  private boolean isXmlSchemaShape(Shape shape) {
    if (!(shape instanceof SchemaShape)) {
      return false;
    }
    SchemaShape schemaShape = (SchemaShape) shape;
    return schemaShape.raw().nonEmpty() && isXmlSchema(schemaShape.raw().value());
  }

  private static Shape getParameterShape(Parameter parameter) {
    Shape shape = parameter.schema();

    if (shape == null && parameter.payloads().size() > 0) {
      shape = parameter.payloads().stream().filter(x -> x.schema() != null).map(Payload::schema).findFirst().orElse(null);
    }

    return shape;
  }
}
