/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import static com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIType.MULTIPART;
import static java.lang.String.format;
import static java.lang.String.join;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON;
import static javax.ws.rs.core.MediaType.APPLICATION_XML;
import static javax.ws.rs.core.MediaType.APPLICATION_FORM_URLENCODED;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA;
import static javax.ws.rs.core.MediaType.TEXT_HTML;
import static javax.ws.rs.core.MediaType.TEXT_PLAIN;
import static org.slf4j.LoggerFactory.getLogger;

import amf.shapes.client.platform.model.domain.AnyShape;
import amf.shapes.client.platform.model.domain.ArrayShape;
import amf.shapes.client.platform.model.domain.FileShape;
import amf.shapes.client.platform.model.domain.NodeShape;
import amf.shapes.client.platform.model.domain.ScalarShape;
import amf.shapes.client.platform.model.domain.SchemaShape;
import amf.shapes.client.platform.model.domain.UnionShape;
import amf.shapes.client.platform.model.domain.Example;
import amf.core.client.platform.model.domain.DataNode;
import amf.core.client.platform.model.domain.Shape;
import amf.core.client.platform.model.domain.PropertyShape;
import amf.core.client.platform.model.domain.ScalarNode;

import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIType;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APIPrimitiveTypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.type.APITypeModel;
import org.slf4j.Logger;

import javax.ws.rs.core.MediaType;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

public class AMFTypeModel extends APITypeModel {

  private static final Logger LOGGER = getLogger(AMFTypeModel.class);
  private static final List<String> ALLOWED_MEDIA_TYPES_FOR_EXAMPLE =
      new ArrayList<String>() {

        {
          add(APPLICATION_JSON);
          add(APPLICATION_XML);
          add(APPLICATION_FORM_URLENCODED);
          add(MULTIPART_FORM_DATA);
          add("text/csv");
          add(TEXT_HTML);
          add(TEXT_PLAIN);
        }
      };

  private AnyShape shape;
  private boolean extractExample;

  public AMFTypeModel(APIPrimitiveTypeModel primitiveTypeModel) {
    super(primitiveTypeModel);
  }

  public AMFTypeModel(AnyShape anyShape, String mediaType, boolean extractExample) {
    this.shape = anyShape;

    this.mediaType = getMediaTypeForStringOrNull(mediaType);
    this.extractExample = extractExample;
    if (extractExample) {
      this.example = buildResponseExample();
    }
    this.enumValues = buildEnumValues();
    this.APITypeSchemaModel = AMFTypeSchemaModel.builder().build(anyShape, this.mediaType);
    this.apiType = buildTypeDefinitionClass(this.mediaType);
    this.displayName = buildDisplayName();
    this.description = buildDescription();
  }

  private String buildDescription() {
    return shape.description().isNullOrEmpty() ? null : shape.description().value();
  }

  private String buildDisplayName() {
    return shape.displayName().isNullOrEmpty() ? null : shape.displayName().value();
  }

  private APIType buildTypeDefinitionClass(MediaType mediaType) {
    if (shape instanceof ScalarShape || shape instanceof FileShape) {
      this.primitiveTypeModel = new AMFPrimitiveTypeModel(shape);
      return APIType.PRIMITIVE;
    }

    else if (MULTIPART_FORM_DATA_TYPE.equals(mediaType)
        && shape instanceof NodeShape
        && !((NodeShape) shape).properties().isEmpty()) {
      return MULTIPART;
    }

    else if (shape instanceof NodeShape || shape instanceof SchemaShape) {
      return APIType.OBJECT_TYPE;
    }

    else if (shape instanceof UnionShape) {
      return APIType.UNION_TYPE;
    }

    else if (shape instanceof ArrayShape) {
      return APIType.ARRAY;
    }

    else if (!shape.and().isEmpty() || !shape.or().isEmpty() || !shape.xone().isEmpty()) {
      return APIType.OBJECT_TYPE;
    }

    // This is for the case that the type is not defined (for instance, when there is an example but not a definition).
    // Also, when the type is defined as 'nil'.
    else {
      return APIType.EMPTY;
    }
  }

  private String buildResponseExample() {
    if (shape.examples().isEmpty() || !isResponseMediaTypeValid()) {
      return null;
    }
    return getExampleFromResponsePayload().orElse(null);
  }

  private boolean isResponseMediaTypeValid() {
    if (this.mediaType == null) {
      LOGGER.warn(format("An exception occurred while extracting and processing sample data example to be auto generated. %s",
                         format("A null output media type has been found and is not "
                             + "allowed to auto generate the sample data based on an example. Allowed media types are: %s\"",
                                join(",", ALLOWED_MEDIA_TYPES_FOR_EXAMPLE))));
      return false;
    }
    if (ALLOWED_MEDIA_TYPES_FOR_EXAMPLE.stream().noneMatch(x -> x.equals(this.mediaType.toString()))) {
      LOGGER.warn(format("An exception occurred while extracting and processing sample data example to be auto generated. %s",
                         format("The following type is not allowed to auto generate "
                             + "the sample data based on an example: %s. Allowed media types are: %s",
                                this.mediaType.toString(), join(",", ALLOWED_MEDIA_TYPES_FOR_EXAMPLE))));
      return false;
    }
    return true;
  }

  private Optional<String> getExampleFromResponsePayload() {
    String example = null;
    Optional<Example> exampleString =
        shape.examples().stream()
            .filter(x -> isYamlExample(x) || isRamlExample(x))
            .findFirst();
    if (exampleString.isPresent()) {
      example = exampleString.get().value().value();
    }
    return Optional.ofNullable(example);
  }

  private boolean isRamlExample(Example example) {
    return example.mediaType().value() == null;
  }

  private boolean isYamlExample(Example example) {
    return example.mediaType().value() != null && example.mediaType().value().equals(this.mediaType.toString());
  }

  private List<String> buildEnumValues() {
    return (!shape.values().isEmpty())
        ? shape.values().stream().map(this::getEnumValue).collect(toList())
        : null;
  }

  @Override
  public List<APIParameterModel> getParts() {
    if (apiType.equals(MULTIPART)) {
      List<APIParameterModel> parts = new ArrayList<>();

      for (PropertyShape s : ((NodeShape) shape).properties()) {
        parts.add(new AMFParameterModel(s, APIParameterType.PART));
      }

      return parts;
    } else {
      return emptyList();
    }
  }

  @Override
  public APITypeModel getInnerType() {
    if (shape instanceof ArrayShape) {
      Shape innerShape = ((ArrayShape) shape).items();

      if (innerShape instanceof AnyShape) {
        return new AMFTypeModel((AnyShape) innerShape, null, this.extractExample);
      } else {
        // Defaulting to string type for shapes that dont implement AnyShape.
        // These shapes can not generate a json/xml schema.
        // TODO: RSDK-8: Recursive shapes can't generate schemas.
        return new AMFTypeModel(new AMFPrimitiveTypeModel(APIPrimitiveType.STRING));
      }
    } else {
      return null;
    }
  }

  @Override
  public List<APITypeModel> getUnionTypes() {
    if (shape instanceof UnionShape) {
      List<APITypeModel> list = new ArrayList<>();

      for (Shape x : ((UnionShape) shape).anyOf()) {
        AMFTypeModel amfTypeParser = new AMFTypeModel((AnyShape) x, null, this.extractExample);
        list.add(amfTypeParser);
      }

      return list;
    } else {
      return emptyList();
    }
  }

  private String getEnumValue(DataNode x) {
    if (x instanceof ScalarNode) {
      return ((ScalarNode) x).value().value();
    }
    throw new IllegalArgumentException("Enum type not supported");
  }
}
