/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.amf;

import amf.apicontract.client.platform.AMFBaseUnitClient;
import amf.apicontract.client.platform.AMFConfiguration;
import amf.apicontract.client.platform.WebAPIConfiguration;
import amf.apicontract.client.platform.model.domain.api.WebApi;
import amf.apicontract.client.scala.UnrecognizedSpecException;
import amf.core.client.platform.AMFParseResult;
import amf.core.client.platform.model.document.BaseUnit;
import amf.core.client.platform.model.document.Document;
import amf.core.client.platform.resource.ClasspathResourceLoader;
import amf.core.client.platform.resource.FileResourceLoader;
import amf.core.client.platform.resource.HttpResourceLoader;
import amf.core.client.platform.validation.AMFValidationReport;
import amf.core.client.platform.validation.AMFValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.InvalidSourceException;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.exception.ModelGenerationException;
import org.apache.commons.lang3.ObjectUtils;
import org.mulesoft.common.client.lexical.Position;
import org.mulesoft.common.client.lexical.PositionRange;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutionException;

import static amf.apicontract.client.platform.WebAPIConfiguration.WebAPI;
import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.lang.Thread.currentThread;
import static java.util.Arrays.asList;
import static java.util.stream.Collectors.joining;
import static org.slf4j.LoggerFactory.getLogger;

/**
 * Builds an API model from an API spec.
 */
public class AMFAPIModelFactory {

  public static final String AMF_VIOLATION_LEVEL = "VIOLATION";
  private static final Logger LOGGER = getLogger(AMFAPIModelFactory.class);
  private static final byte[] RAML_FILE_START = "#%RAML 0.8".getBytes(StandardCharsets.UTF_8);
  private final AMFBaseUnitClient baseUnitClient;

  public AMFAPIModelFactory() {
    this(Collections.emptyList());
  }

  public AMFAPIModelFactory(List<String> dialectUrls) {
    AMFConfiguration amfWebApiConfig = WebAPI();
    amfWebApiConfig = configureResourceLoaders(amfWebApiConfig);
    amfWebApiConfig = configureDialects(amfWebApiConfig, dialectUrls);
    baseUnitClient = amfWebApiConfig.baseUnitClient();
  }

  /**
   * Builds an API model from an API spec file.
   *
   * @param apiSpecPath the input spec file
   * @param skipValidations whether to run AMF validations
   * @return the model
   */
  public AMFAPIModel build(Path apiSpecPath, boolean skipValidations) throws ModelGenerationException {
    return build(getWebApi(apiSpecPath, skipValidations));
  }

  /**
   * Builds an API model from a string containing an API spec.
   *
   * @param apiSpec the string with an API spec
   * @param skipValidations whether to run AMF validations
   * @return the model
   */
  public AMFAPIModel build(String apiSpec, boolean skipValidations) throws ModelGenerationException {
    return build(getWebApi(apiSpec, skipValidations));
  }

  private static AMFAPIModel build(ParseResult parseResult) throws ModelGenerationException {
    return new AMFAPIModel(parseResult.webApi, parseResult.isRaml);
  }

  private static AMFConfiguration configureResourceLoaders(AMFConfiguration amfConfiguration) {
    return amfConfiguration.withResourceLoaders(asList(new FileResourceLoader(), new HttpResourceLoader(),
                                                       new ClasspathResourceLoader()));
  }

  private static AMFConfiguration configureDialects(AMFConfiguration amfConfiguration, List<String> dialectUrls) {
    for (String dialectUrl : dialectUrls) {
      try {
        amfConfiguration = amfConfiguration.withDialect(dialectUrl).get();
      } catch (ExecutionException e) {
        throw new RuntimeException("Could not register dialect '" + dialectUrl + "': " + e.getCause(), e.getCause());
      } catch (InterruptedException e) {
        currentThread().interrupt();
        throw new RuntimeException("Could not register dialect '" + dialectUrl + "': " + e, e);
      }
    }
    return amfConfiguration;
  }

  private ParseResult getWebApi(Path path, boolean skipValidations)
      throws InvalidSourceException {
    if (Files.notExists(path)) {
      throw new InvalidSourceException(format("Spec file [%s] does not exist", path.toAbsolutePath()));
    }

    if (isRaml08(path)) {
      throw new InvalidSourceException("RAML 0.8 is not supported.");
    }

    try {
      return getWebApi(baseUnitClient.parse("file://" + path.toAbsolutePath()).get(), skipValidations);
    } catch (InterruptedException | ExecutionException e) {
      throw new InvalidSourceException("Error parsing spec: " + e.getMessage(), e);
    } catch (UnrecognizedSpecException e) {
      throw new InvalidSourceException("Error parsing spec: " + e.getMessage() + ". Supported encoding: UTF-8.", e);
    }
  }

  private ParseResult getWebApi(String content, boolean skipValidations) throws InvalidSourceException {
    try {
      return getWebApi(baseUnitClient.parseContent(content).get(), skipValidations);
    } catch (InterruptedException | ExecutionException e) {
      throw new InvalidSourceException("Error parsing spec: " + e.getMessage(), e);
    } catch (UnrecognizedSpecException e) {
      throw new InvalidSourceException("Error parsing spec: " + e.getMessage() + ". Supported encoding: UTF-8.", e);
    }
  }

  private ParseResult getWebApi(AMFParseResult amfParseResult, boolean skipValidations) throws InvalidSourceException {
    boolean isRaml = amfParseResult.sourceSpec() != null && amfParseResult.sourceSpec().isRaml();
    AMFConfiguration specificConfiguration = WebAPIConfiguration.fromSpec(amfParseResult.sourceSpec());

    if (!amfParseResult.conforms()) {
      throw new InvalidSourceException("The provided API specification has errors.\n" +
          amfParseResult.results().stream()
              .map(AMFAPIModelFactory::validationResultMessage)
              .collect(joining(lineSeparator())));
    }

    AMFBaseUnitClient specBaseUnitClient = specificConfiguration.baseUnitClient();
    BaseUnit baseUnit = specBaseUnitClient.transform(amfParseResult.baseUnit()).baseUnit();

    Document document;
    if (baseUnit instanceof Document) {
      document = (Document) baseUnit;
    } else {
      throw new InvalidSourceException("Document type not supported: " + baseUnit.getClass());
    }

    if (!skipValidations) {
      validateDocument(specBaseUnitClient, document);
    }

    return new ParseResult((WebApi) document.encodes(), isRaml);
  }

  private static String validationResultMessage(AMFValidationResult amfValidationResult) {
    return amfValidationResult.message().replace("\n", "\\n") + amfValidationResult.location().map(loc -> {
      PositionRange position = amfValidationResult.position();
      return ", at " + loc + ':' + position.start() + ' ' + position.end();
    }).orElse("");
  }

  private void validateDocument(AMFBaseUnitClient client, Document document) throws InvalidSourceException {
    AMFValidationReport report;

    try {
      report = client.validate(document).get();
    } catch (Exception e) {
      String message = format("Error validating the spec [%s], detail: %s.", document.location(), e.getMessage());
      throw new InvalidSourceException(message);
    }

    if (!report.conforms()) {
      printErrors(report);

      String errorMessage = format("The API spec [%s] is invalid, please fix it (or ignore it).", document.location());
      throw new InvalidSourceException(errorMessage);
    }
  }

  private static void printErrors(AMFValidationReport report) {
    for (AMFValidationResult result : report.results()) {
      if (result.severityLevel().equalsIgnoreCase(AMF_VIOLATION_LEVEL)) {
        LOGGER.error(report.toString());
      }
    }
  }

  private static boolean isRaml08(Path specPath) throws InvalidSourceException {
    try (InputStream i = Files.newInputStream(specPath)) {
      byte[] buf = new byte[RAML_FILE_START.length];
      return i.read(buf) == RAML_FILE_START.length && Arrays.equals(buf, RAML_FILE_START);
    } catch (IOException e) {
      throw new InvalidSourceException("Could not check whether API spec [" + specPath + "] is a RAML file: " + e, e);
    }
  }

  private static class ParseResult {

    final WebApi webApi;
    final boolean isRaml;

    private ParseResult(WebApi webApi, boolean isRaml) {
      this.webApi = webApi;
      this.isRaml = isRaml;
    }
  }
}
