/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.HEADER;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.QUERY;
import static com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType.URI;
import static java.util.Objects.requireNonNull;

import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.ParameterBinding;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.Trigger;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.TriggerParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.ParameterDataType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterBindingDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerWatermarkDescriptor;

import java.util.LinkedList;
import java.util.List;

public class ConnectorTriggerBuilder {

  public List<Trigger> buildTriggers(ConnectorDescriptor connectorDescriptor, List<ConnectorOperation> operations) {
    final List<Trigger> triggers = new LinkedList<>();

    if (connectorDescriptor.getTriggers() != null) {
      for (TriggerDescriptor triggerDescriptor : connectorDescriptor.getTriggers()) {
        ConnectorOperation operation = operations.stream()
            .filter(x -> x.getPath().equalsIgnoreCase(triggerDescriptor.getPath()))
            .filter(x -> x.getHttpMethod().equalsIgnoreCase(triggerDescriptor.getMethod().getName()))
            .findFirst().orElse(null);

        requireNonNull(operation);
        triggers.add(buildTrigger(operation, triggerDescriptor));
      }
    }

    return triggers;
  }

  public Trigger buildTrigger(ConnectorOperation operation, TriggerDescriptor triggerDescriptor) {
    return new Trigger(triggerDescriptor.getName(),
                       triggerDescriptor.getDisplayName(),
                       triggerDescriptor.getDescription(),
                       triggerDescriptor.getItemsExpression(),
                       triggerDescriptor.getIdentityExpression(),
                       getRequestBodyExpression(triggerDescriptor),
                       getWatermarkExpression(triggerDescriptor),
                       buildWatermarkType(triggerDescriptor.getWatermark()),
                       buildParameters(triggerDescriptor.getParameters()),
                       buildParameterBindings(triggerDescriptor.getParameterBindings()),
                       operation);
  }

  private String getWatermarkExpression(TriggerDescriptor triggerDescriptor) {
    if (triggerDescriptor == null || triggerDescriptor.getWatermark() == null) {
      return null;
    }

    return triggerDescriptor.getWatermark().getExtraction();
  }

  private String getRequestBodyExpression(TriggerDescriptor triggerDescriptor) {
    if (triggerDescriptor == null || triggerDescriptor.getParameterBindings() == null) {
      return null;
    }

    return triggerDescriptor.getParameterBindings().getRequestBodyExpression();
  }

  private ParameterDataType buildWatermarkType(TriggerWatermarkDescriptor watermarkDescriptor) {
    if (watermarkDescriptor == null || watermarkDescriptor.getDataType() == null) {
      return ParameterDataType.STRING;
    }
    return ParameterDataType.forName(watermarkDescriptor.getDataType().getName());
  }

  private List<ParameterBinding> buildParameterBindings(TriggerParameterBindingsDescriptor parameterBindings) {
    if (parameterBindings == null) {
      return null;
    }

    final List<ParameterBinding> parameterBindingList = new LinkedList<>();

    if (parameterBindings.getUriParameters() != null) {
      for (TriggerParameterBindingDescriptor uriParam : parameterBindings.getUriParameters()) {
        parameterBindingList.add(buildParameterBinding(uriParam, URI));
      }
    }

    if (parameterBindings.getQueryParameters() != null) {
      for (TriggerParameterBindingDescriptor queryParam : parameterBindings.getQueryParameters()) {
        parameterBindingList.add(buildParameterBinding(queryParam, QUERY));
      }
    }

    if (parameterBindings.getHeaders() != null) {
      for (TriggerParameterBindingDescriptor header : parameterBindings.getHeaders()) {
        parameterBindingList.add(buildParameterBinding(header, HEADER));
      }
    }

    return parameterBindingList;
  }

  private ParameterBinding buildParameterBinding(TriggerParameterBindingDescriptor descriptor,
                                                 com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType parameterType) {
    return new ParameterBinding(parameterType, descriptor.getName(), descriptor.getValue());
  }

  private List<TriggerParameter> buildParameters(List<TriggerParameterDescriptor> parameterDescriptors) {
    final List<TriggerParameter> parameters = new LinkedList<>();

    for (TriggerParameterDescriptor parameterDescriptor : parameterDescriptors) {
      parameters.add(buildParameter(parameterDescriptor));
    }

    return parameters;
  }

  private TriggerParameter buildParameter(TriggerParameterDescriptor parameterDescriptor) {
    return new TriggerParameter(
                                parameterDescriptor.getName(),
                                parameterDescriptor.getDisplayName(),
                                ParameterDataType.forName(parameterDescriptor.getDataType().getName()),
                                parameterDescriptor.getDescription(),
                                parameterDescriptor.isRequired());
  }
}
