/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.connectormodel.builder;

import static com.mulesoft.connectivity.rest.commons.internal.util.RestSdkUtils.isNotBlank;
import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_XML_TYPE;
import com.mulesoft.connectivity.rest.sdk.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.PartParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.ArrayTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.EmptyTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.MultipartTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.ObjectTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.UnionTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.XmlTypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.PartDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIPrimitiveTypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APITypeModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.parser.TypeSchemaPool;

import java.util.ArrayList;
import java.util.List;

import javax.ws.rs.core.MediaType;

public class ConnectorTypeDefinitionBuilder {

  private final ConnectorTypeSchemaBuilder typeSchemaBuilder;
  private final ConnectorParameterBuilder parameterBuilder;

  public ConnectorTypeDefinitionBuilder(TypeSchemaPool typeSchemaPool) {
    this.typeSchemaBuilder = new ConnectorTypeSchemaBuilder(typeSchemaPool);
    this.parameterBuilder = new ConnectorParameterBuilder(this);
  }

  public TypeDefinition buildTypeDefinition(APITypeModel apiTypeModel, List<PartDescriptor> partDescriptors)
      throws ModelGenerationException {
    return buildTypeDefinition(apiTypeModel, partDescriptors, apiTypeModel.getMediaType());
  }

  public TypeDefinition buildTypeDefinition(APITypeModel apiTypeModel, List<PartDescriptor> partDescriptors, MediaType mediaType)
      throws ModelGenerationException {

    TypeDefinition.Builder typeDefinitionBuilder =
        TypeDefinition.builder(mediaType,
                               typeSchemaBuilder.buildTypeSchema(apiTypeModel.getAPITypeSchemaModel()),
                               apiTypeModel.getEnumValues(),
                               apiTypeModel.getExample());

    if (isNotBlank(apiTypeModel.getDisplayName())) {
      typeDefinitionBuilder.withDisplayName(apiTypeModel.getDisplayName());
    }

    if (isNotBlank(apiTypeModel.getDescription())) {
      typeDefinitionBuilder.withDescription(apiTypeModel.getDescription());
    }

    Class<?> typeDefinitionClass = apiTypeModel.getTypeDefinitionClass();

    if (typeDefinitionClass == null) {
      return null;
    } else if (typeDefinitionClass.equals(EmptyTypeDefinition.class)) {
      return typeDefinitionBuilder.buildEmptyTypeDefinition();
    } else if (typeDefinitionClass.equals(PrimitiveTypeDefinition.class)) {
      return typeDefinitionBuilder.buildPrimitiveTypeDefinition(buildPrimitiveType(apiTypeModel.getPrimitiveTypeModel()));
    } else if (typeDefinitionClass.equals(ObjectTypeDefinition.class)) {
      return typeDefinitionBuilder.buildObjectTypeDefinition();
    } else if (typeDefinitionClass.equals(ArrayTypeDefinition.class)) {
      MediaType innerMediaType =
          apiTypeModel.getInnerType().getMediaType() != null ? apiTypeModel.getInnerType().getMediaType() : mediaType;
      return typeDefinitionBuilder.buildArrayType(buildTypeDefinition(apiTypeModel.getInnerType(), null, innerMediaType));
    } else if (typeDefinitionClass.equals(UnionTypeDefinition.class)) {
      List<TypeDefinition> unionTypes = new ArrayList<>();
      for (APITypeModel typeModel : apiTypeModel.getUnionTypes()) {
        TypeDefinition typeDefinition = buildTypeDefinition(typeModel, null, mediaType);
        unionTypes.add(typeDefinition);
      }
      return typeDefinitionBuilder.buildUnionTypeDefinition(unionTypes);
    } else if (typeDefinitionClass.equals(MultipartTypeDefinition.class)) {
      List<PartParameter> parts = new ArrayList<>();
      for (APIParameterModel apiParameterModel : apiTypeModel.getParts()) {
        PartDescriptor partDescriptor = null;
        if (partDescriptors != null) {
          partDescriptor = partDescriptors.stream()
              .filter(x -> x.getPartName().equals(apiParameterModel.getExternalName()))
              .findFirst().orElse(null);
        }
        PartParameter partParameter = parameterBuilder.buildPartParameter(apiParameterModel, partDescriptor);
        parts.add(partParameter);
      }
      return typeDefinitionBuilder.buildMultipartTypeDefinition(parts);
    } else {
      return null;
    }
  }

  public TypeDefinition buildTypeDefinition(String rawSchema, MediaType mediaType) throws ModelGenerationException {
    TypeSchema typeSchema = typeSchemaBuilder.buildTypeSchema(rawSchema);

    if (mediaType == null) {
      mediaType = typeSchema instanceof XmlTypeSchema ? APPLICATION_XML_TYPE : APPLICATION_JSON_TYPE;
    }

    return TypeDefinition.builder(mediaType, typeSchema, null, null).buildObjectTypeDefinition();
  }

  private PrimitiveTypeDefinition.PrimitiveType buildPrimitiveType(APIPrimitiveTypeModel primitiveTypeModel) {
    return primitiveTypeModel.getPrimitiveType();
  }

}
