/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser;

import amf.client.model.domain.DialectDomainElement;

import com.mulesoft.connectivity.rest.sdk.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.*;

public class DescriptorOperationParser {

  private static final String API_CONTRACT_SUPPORTED_OPERATION =
      "http://a.ml/vocabularies.rest.sdk/apiContract#supportedOperation";

  private static final String REST_SDK_OPERATION_NAME = "http://a.ml/vocabularies/rest-sdk#operationName";
  private static final String REST_SDK_OPERATION_DESCRIPTION = "http://a.ml/vocabularies/rest-sdk#operationDescription";
  private static final String REST_SDK_DEFAULT_INPUT_MEDIA_TYPE = "http://a.ml/vocabularies/rest-sdk#inputMediaType";
  private static final String REST_SDK_DEFAULT_OUTPUT_MEDIA_TYPE = "http://a.ml/vocabularies/rest-sdk#outputMediaType";
  private static final String REST_SDK_INPUT_TYPE = "http://a.ml/vocabularies/rest-sdk#inputType";
  private static final String REST_SDK_OUTPUT_TYPE = "http://a.ml/vocabularies/rest-sdk#outputType";
  private static final String REST_SDK_PAGINATION = "http://a.ml/vocabularies/rest-sdk#paginationReference";
  private static final String REST_SDK_VOID_OPERATION = "http://a.ml/vocabularies/rest-sdk#voidOperation";
  private static final String QUERY_PARAM_ARRAY_FORMAT = "http://a.ml/vocabularies/rest-sdk#queryParameterArrayFormat";

  private final DescriptorRequestParser descriptorRequestParser = new DescriptorRequestParser();
  private final DescriptorEncodesParser descriptorEncodesParser = new DescriptorEncodesParser();


  public List<OperationDescriptor> parseOperations(DialectDomainElement endpointElement, Path descriptorFilePath)
      throws ModelGenerationException {

    List<OperationDescriptor> operationDescriptors = new ArrayList<>();

    for (DialectDomainElement x : getSupportedOperations(endpointElement)) {
      operationDescriptors.add(parseOperation(x, descriptorFilePath));
    }

    return operationDescriptors;
  }

  private OperationDescriptor parseOperation(DialectDomainElement operationElement, Path descriptorFilePath)
      throws ModelGenerationException {
    Path descriptorDirectory = descriptorFilePath.getParent();



    return new OperationDescriptor(parseApiContractMethod(operationElement),
                                   getName(operationElement),
                                   getDescription(operationElement),
                                   getDefaultInputMediaType(operationElement),
                                   getDefaultOutputMediaType(operationElement),
                                   descriptorRequestParser.parseRequest(operationElement, descriptorDirectory),
                                   parseNullableIgnore(operationElement),
                                   parseAlternativeBaseUri(operationElement),
                                   getPagination(operationElement),
                                   descriptorEncodesParser.parseSkipOutputTypeValidation(operationElement),
                                   parseVoidOperation(operationElement),
                                   getInputTypeSchema(operationElement, descriptorDirectory),
                                   getOutputTypeSchema(operationElement, descriptorDirectory),
                                   getQueryParamArrayFormat(operationElement));
  }

  private List<DialectDomainElement> getSupportedOperations(DialectDomainElement endpointElement) {
    return endpointElement.getObjectPropertyUri(API_CONTRACT_SUPPORTED_OPERATION);
  }

  private Boolean parseVoidOperation(DialectDomainElement operationElement) {
    return parseSingleBooleanOrNullProperty(operationElement, REST_SDK_VOID_OPERATION);
  }

  private String getDescription(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, REST_SDK_OPERATION_DESCRIPTION);
  }

  private String getName(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, REST_SDK_OPERATION_NAME);
  }

  private String getDefaultInputMediaType(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, REST_SDK_DEFAULT_INPUT_MEDIA_TYPE);
  }

  private String getInputTypeSchema(DialectDomainElement operationElement, Path descriptorDirectory)
      throws ModelGenerationException {
    String inputSchemaPath = parseSingleStringProperty(operationElement, REST_SDK_INPUT_TYPE);
    return getSchemaContent(descriptorDirectory, inputSchemaPath);
  }

  private String getOutputTypeSchema(DialectDomainElement operationElement, Path descriptorDirectory)
      throws ModelGenerationException {
    String outputSchemaPath = parseSingleStringProperty(operationElement, REST_SDK_OUTPUT_TYPE);
    return getSchemaContent(descriptorDirectory, outputSchemaPath);
  }

  private String getDefaultOutputMediaType(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, REST_SDK_DEFAULT_OUTPUT_MEDIA_TYPE);
  }

  private String getPagination(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, REST_SDK_PAGINATION);
  }

  private String getQueryParamArrayFormat(DialectDomainElement operationElement) {
    return parseSingleStringProperty(operationElement, QUERY_PARAM_ARRAY_FORMAT);
  }
}
