/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser;

import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.HttpMethodDescriptor.forName;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.nio.file.Files.readAllBytes;
import static org.apache.commons.lang3.StringUtils.isBlank;
import com.mulesoft.connectivity.rest.sdk.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.DataTypeDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.HttpMethodDescriptor;

import java.io.IOException;
import java.nio.file.Path;
import java.util.List;

import amf.client.model.domain.DialectDomainElement;

public final class DescriptorParserUtils {

  private static final String CORE_NAME = "http://a.ml/vocabularies.rest.sdk/core#name";
  private static final String CORE_DESCRIPTION = "http://a.ml/vocabularies.rest.sdk/core#description";
  private static final String REST_SDK_IGNORE = "http://a.ml/vocabularies/rest-sdk#ignore";
  private static final String REST_SDK_BASE_URI_REFERENCE = "http://a.ml/vocabularies/rest-sdk#baseUriReference";
  private static final String REST_SDK_FRIENDLY_NAME = "http://a.ml/vocabularies/rest-sdk#friendlyName";
  private static final String REST_SDK_REQUIRED = "http://a.ml/vocabularies/rest-sdk#required";
  private static final String API_CONTRACT_PATH = "http://a.ml/vocabularies.rest.sdk/apiContract#path";
  private static final String API_CONTRACT_METHOD = "http://a.ml/vocabularies.rest.sdk/apiContract#method";
  private static final String API_CONTRACT_PARAM_NAME = "http://a.ml/vocabularies.rest.sdk/apiContract#paramName";
  private static final String AML_REST_SDK_PARAMETER_TYPE = "http://a.ml/vocabularies/rest-sdk#parameterType";

  private DescriptorParserUtils() {}

  public static Boolean parseNullableIgnore(DialectDomainElement element) {
    return parseSingleBooleanOrNullProperty(element, REST_SDK_IGNORE);
  }

  public static Boolean parseIgnore(DialectDomainElement element) {
    return parseSingleBooleanProperty(element, REST_SDK_IGNORE);
  }

  public static boolean parseRequired(DialectDomainElement element) {
    return parseSingleBooleanProperty(element, REST_SDK_REQUIRED);
  }

  public static String parseAlternativeBaseUri(DialectDomainElement element) {
    return parseSingleStringProperty(element, REST_SDK_BASE_URI_REFERENCE);
  }

  public static String parseApiContractParamName(DialectDomainElement element) {
    return parseSingleStringProperty(element, API_CONTRACT_PARAM_NAME);
  }

  public static String parseCoreName(DialectDomainElement element) {
    return parseSingleStringProperty(element, CORE_NAME);
  }

  public static String parseCoreDescription(DialectDomainElement element) {
    return parseSingleStringProperty(element, CORE_DESCRIPTION);
  }

  public static String parseFriendlyName(DialectDomainElement element) {
    return parseSingleStringProperty(element, REST_SDK_FRIENDLY_NAME);
  }

  public static String parseApiContractPath(DialectDomainElement element) {
    return parseSingleStringProperty(element, API_CONTRACT_PATH);
  }

  public static String parseApiContractMethod(DialectDomainElement element) {
    return parseSingleStringProperty(element, API_CONTRACT_METHOD);
  }

  public static HttpMethodDescriptor parseApiContractMethodAsMethodDescriptor(DialectDomainElement element) {
    return forName(parseApiContractMethod(element));
  }

  public static DataTypeDescriptor parseParameterType(DialectDomainElement parameterDescriptor) {
    return DataTypeDescriptor.forName(parseSingleStringProperty(parameterDescriptor, AML_REST_SDK_PARAMETER_TYPE));
  }

  public static String parseSingleStringProperty(DialectDomainElement element, String propertyFQN) {
    List<Object> propertyInstances = element.getScalarByPropertyUri(propertyFQN);
    return propertyInstances.isEmpty() ? null : propertyInstances.get(0).toString();
  }

  public static boolean parseSingleBooleanProperty(DialectDomainElement element, String propertyFQN) {
    List<Object> propertyInstances = element.getScalarByPropertyUri(propertyFQN);
    return !propertyInstances.isEmpty() && Boolean.parseBoolean(propertyInstances.get(0).toString());
  }

  public static Boolean parseSingleBooleanOrNullProperty(DialectDomainElement element, String propertyFQN) {
    List<Object> propertyInstances = element.getScalarByPropertyUri(propertyFQN);
    return propertyInstances.isEmpty() ? null : Boolean.parseBoolean(propertyInstances.get(0).toString());
  }

  public static DialectDomainElement getSingleObjectProperty(DialectDomainElement element, String propertyFQN) {
    List<DialectDomainElement> propertyInstances = element.getObjectPropertyUri(propertyFQN);
    return propertyInstances.isEmpty() ? null : propertyInstances.get(0);
  }

  public static String getSchemaContent(Path descriptorDirectory, String inputSchemaPathString) throws ModelGenerationException {
    if (isBlank(inputSchemaPathString)) {
      return null;
    }

    Path schemaPath = descriptorDirectory.resolve(inputSchemaPathString);

    try {
      byte[] encoded = readAllBytes(schemaPath);
      return new String(encoded, UTF_8);
    } catch (IOException e) {
      throw new ModelGenerationException("Could not read schema file: " + schemaPath, e);
    }
  }

}
