/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser;

import static java.lang.String.format;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.getSingleObjectProperty;

import com.mulesoft.connectivity.rest.sdk.exception.ModelGenerationException;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.RequestDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;

import com.google.common.collect.ImmutableMap;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import amf.client.model.domain.DialectDomainElement;

public class DescriptorRequestParser {

  private static final Map<String, String> AMF_PARAMETER_TYPE = ImmutableMap.<String, String>builder()
      .put(ParameterType.URI.getName(), "uriParameter")
      .put(ParameterType.QUERY.getName(), "queryString")
      .put(ParameterType.HEADER.getName(), "header")
      .build();

  private static final String API_CONTRACT_EXPECTS = "http://a.ml/vocabularies.rest.sdk/apiContract#expects";
  private static final String API_CONTRACT_PARAMETER_NAMES = "http://a.ml/vocabularies.rest.sdk/apiContract#%s";
  private static final String REST_SDK_PART = "http://a.ml/vocabularies/rest-sdk#part";

  private static final DescriptorParameterParser descriptorParameterParser = new DescriptorParameterParser();
  private static final DescriptorPartParser descriptorPartParser = new DescriptorPartParser();

  public RequestDescriptor parseRequest(DialectDomainElement operationElement, Path descriptorFilePath)
      throws ModelGenerationException {
    DialectDomainElement expectsElement = parseExpectsElement(operationElement);

    if (expectsElement == null) {
      return null;
    }

    return new RequestDescriptor(
                                 descriptorParameterParser
                                     .parseParameters(getParametersByType(expectsElement, ParameterType.HEADER)),
                                 descriptorParameterParser
                                     .parseParameters(getParametersByType(expectsElement, ParameterType.QUERY)),
                                 descriptorParameterParser
                                     .parseParameters(getParametersByType(expectsElement, ParameterType.URI)),
                                 descriptorPartParser.parseParts(getParts(expectsElement), descriptorFilePath));
  }


  private List<DialectDomainElement> getParametersByType(DialectDomainElement expects, ParameterType parameterType) {
    String amfParameterType = format(API_CONTRACT_PARAMETER_NAMES, AMF_PARAMETER_TYPE.get(parameterType.getName()));
    return new ArrayList<>(expects.getObjectPropertyUri(amfParameterType));
  }

  private List<DialectDomainElement> getParts(DialectDomainElement expects) {
    return new ArrayList<>(expects.getObjectPropertyUri(REST_SDK_PART));
  }

  private DialectDomainElement parseExpectsElement(DialectDomainElement operationElement) {
    return getSingleObjectProperty(operationElement, API_CONTRACT_EXPECTS);
  }
}
