/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser;

import static java.util.Collections.emptySet;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseSingleStringProperty;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TestConnectionDescriptor;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.validation.constraints.NotNull;

import amf.client.model.domain.DialectDomainElement;
import amf.core.model.domain.AmfArray;
import amf.core.model.domain.AmfElement;
import amf.core.parser.Value;
import scala.collection.Iterator;

public class DescriptorTestConnectionParser {

  private static final String REST_SDK_TEST_CONNECTION = "http://a.ml/vocabularies/rest-sdk#testConnection";
  private static final String REST_SDK_TEST_CONNECTION_PATH = "http://a.ml/vocabularies/rest-sdk#testConnectionPath";
  private static final String REST_SDK_TEST_CONNECTION_METHOD = "http://a.ml/vocabularies/rest-sdk#testConnectionMethod";
  private static final String REST_SDK_TEST_CONNECTION_STATUS_CODES =
      "http://a.ml/vocabularies/rest-sdk#testConnectionValidStatusCodes";
  private static final String REST_SDK_TEST_CONNECTION_MEDIA_TYPE =
      "http://a.ml/vocabularies/rest-sdk#testConnectionMediaType";

  private static final DescriptorTestConnectionResponseValidationParser responseValidationParser =
      new DescriptorTestConnectionResponseValidationParser();

  public TestConnectionDescriptor parseTestConnection(DialectDomainElement parentElement) {
    DialectDomainElement testConnectionElement = parentElement
        .getObjectPropertyUri(REST_SDK_TEST_CONNECTION)
        .stream().findFirst().orElse(null);

    if (testConnectionElement == null) {
      return null;
    }

    return new TestConnectionDescriptor(
                                        getPath(testConnectionElement),
                                        getMethod(testConnectionElement),
                                        getValidStatusCodes(testConnectionElement),
                                        getMediaType(testConnectionElement),
                                        responseValidationParser.parseResponseValidations(testConnectionElement));
  }

  private String getPath(@NotNull DialectDomainElement testConnectionElement) {
    return parseSingleStringProperty(testConnectionElement, REST_SDK_TEST_CONNECTION_PATH);
  }

  private String getMethod(@NotNull DialectDomainElement testConnectionElement) {
    return parseSingleStringProperty(testConnectionElement, REST_SDK_TEST_CONNECTION_METHOD);
  }

  private Set<String> getValidStatusCodes(@NotNull DialectDomainElement testConnectionElement) {
    List<Object> validStatusCodes = testConnectionElement.getScalarByPropertyUri(REST_SDK_TEST_CONNECTION_STATUS_CODES);

    if (!validStatusCodes.isEmpty() &&
        validStatusCodes.get(0) instanceof Value &&
        ((Value) validStatusCodes.get(0)).value() instanceof AmfArray) {
      Set<String> result = new HashSet<>();

      AmfArray amfArray = (AmfArray) (((Value) validStatusCodes.get(0)).value());
      Iterator<AmfElement> iterator = amfArray.values().iterator();
      while (iterator.hasNext()) {
        result.add(iterator.next().toString());
      }

      return result;
    }

    return emptySet();
  }

  private String getMediaType(@NotNull DialectDomainElement testConnectionElement) {
    return parseSingleStringProperty(testConnectionElement, REST_SDK_TEST_CONNECTION_MEDIA_TYPE);
  }
}
