/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser;

import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.getSingleObjectProperty;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseApiContractMethodAsMethodDescriptor;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseApiContractPath;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseCoreDescription;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseCoreName;
import static com.mulesoft.connectivity.rest.sdk.internal.descriptor.parser.DescriptorParserUtils.parseFriendlyName;
import static java.util.Collections.emptyList;

import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ExpressionDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterBindingsDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.TriggerWatermarkDescriptor;

import java.util.LinkedList;
import java.util.List;

import amf.client.model.domain.DialectDomainElement;

public class DescriptorTriggerParser {

  private static final String AML_REST_SDK_TRIGGERS = "http://a.ml/vocabularies/rest-sdk#triggers";
  private static final String AML_REST_SDK_TRIGGER_PARAMETERS = "http://a.ml/vocabularies/rest-sdk#triggerParameters";
  private static final String AML_REST_SDK_TRIGGER_WATERMARK = "http://a.ml/vocabularies/rest-sdk#triggerWatermark";

  private static final String AML_REST_SDK_TRIGGER_BINDINGS = "http://a.ml/vocabularies/rest-sdk#triggerParameterBindings";

  private static final String AML_REST_SDK_TRIGGER_ITEMS = "http://a.ml/vocabularies/rest-sdk#triggerItems";
  private static final String AML_REST_SDK_TRIGGER_IDENTITY = "http://a.ml/vocabularies/rest-sdk#triggerIdentity";
  private static final String AML_REST_SDK_TRIGGER_ITEMS_EXTRACTION =
      "http://a.ml/vocabularies/rest-sdk#triggerItemsExtraction";
  private static final String AML_REST_SDK_TRIGGER_IDENTITY_EXTRACTION =
      "http://a.ml/vocabularies/rest-sdk#triggerIdentityExtraction";

  private static final DescriptorTriggerWatermarkParser watermarkParser = new DescriptorTriggerWatermarkParser();
  private static final DescriptorTriggerParametersParser parameterParser = new DescriptorTriggerParametersParser();
  private static final DescriptorTriggerParameterBindingParser bindingParser = new DescriptorTriggerParameterBindingParser();
  private static final DescriptorExpressionParser expressionParser = new DescriptorExpressionParser();

  public List<TriggerDescriptor> parseTriggers(DialectDomainElement encodes) {
    List<DialectDomainElement> triggerDescriptors = encodes.getObjectPropertyUri(AML_REST_SDK_TRIGGERS);

    if (triggerDescriptors == null || triggerDescriptors.isEmpty()) {
      return emptyList();
    }

    List<TriggerDescriptor> result = new LinkedList<>();
    for (DialectDomainElement triggerDescriptor : triggerDescriptors) {
      result.add(parseTrigger(triggerDescriptor));
    }
    return result;
  }

  private TriggerDescriptor parseTrigger(DialectDomainElement triggerDescriptor) {
    return new TriggerDescriptor(
                                 parseCoreName(triggerDescriptor),
                                 parseApiContractPath(triggerDescriptor),
                                 parseApiContractMethodAsMethodDescriptor(triggerDescriptor),
                                 parseFriendlyName(triggerDescriptor),
                                 parseCoreDescription(triggerDescriptor),
                                 parseParameters(triggerDescriptor),
                                 parseParameterBindings(triggerDescriptor),
                                 parseWatermark(triggerDescriptor),
                                 parseItemsExpression(triggerDescriptor),
                                 parseIdentityExpression(triggerDescriptor));
  }

  private String parseIdentityExpression(DialectDomainElement triggerDescriptor) {
    DialectDomainElement itemsDescriptor = getSingleObjectProperty(triggerDescriptor, AML_REST_SDK_TRIGGER_IDENTITY);

    if (itemsDescriptor == null)
      return null;

    ExpressionDescriptor expressionDescriptor =
        expressionParser.parseExpression(
                                         getSingleObjectProperty(itemsDescriptor, AML_REST_SDK_TRIGGER_IDENTITY_EXTRACTION));

    return expressionDescriptor.getExpression();
  }

  private String parseItemsExpression(DialectDomainElement triggerDescriptor) {
    DialectDomainElement itemsDescriptor = getSingleObjectProperty(triggerDescriptor, AML_REST_SDK_TRIGGER_ITEMS);

    if (itemsDescriptor == null)
      return null;

    ExpressionDescriptor expressionDescriptor =
        expressionParser.parseExpression(
                                         getSingleObjectProperty(itemsDescriptor, AML_REST_SDK_TRIGGER_ITEMS_EXTRACTION));

    return expressionDescriptor.getExpression();
  }

  private TriggerWatermarkDescriptor parseWatermark(DialectDomainElement triggerDescriptor) {
    DialectDomainElement watermarkDescriptor = getSingleObjectProperty(triggerDescriptor, AML_REST_SDK_TRIGGER_WATERMARK);
    return watermarkParser.parseWatermark(watermarkDescriptor);
  }

  private List<TriggerParameterDescriptor> parseParameters(DialectDomainElement triggerDescriptor) {
    return parameterParser.parseParameters(triggerDescriptor.getObjectPropertyUri(AML_REST_SDK_TRIGGER_PARAMETERS));
  }

  private TriggerParameterBindingsDescriptor parseParameterBindings(DialectDomainElement triggerDescriptor) {
    return bindingParser.parseParameterBindings(getSingleObjectProperty(triggerDescriptor, AML_REST_SDK_TRIGGER_BINDINGS));
  }
}
