/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.templating;

import static com.google.common.base.CaseFormat.UPPER_CAMEL;
import static javax.lang.model.element.Modifier.FINAL;
import static javax.lang.model.element.Modifier.PRIVATE;
import static javax.lang.model.element.Modifier.STATIC;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

import com.mulesoft.connectivity.rest.sdk.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.trigger.ParameterDataType;

import com.google.common.base.CaseFormat;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeSpec;

import java.nio.file.Path;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.util.Optional;

import javax.lang.model.element.Modifier;


public abstract class JavaTemplateEntity extends TemplateEntity {

  protected static final String VALUE_MEMBER = "value";
  protected static final String NAME_MEMBER = "name";

  private static final String NULL = "null";

  protected final Path outputDir;
  protected final ConnectorModel connectorModel;
  protected final RestSdkRunConfiguration runConfiguration;

  public JavaTemplateEntity(Path outputDir, ConnectorModel connectorModel, RestSdkRunConfiguration runConfiguration) {
    this.outputDir = outputDir;
    this.connectorModel = connectorModel;
    this.runConfiguration = runConfiguration;
  }

  protected void writeClassToFile(TypeSpec clazz, String targetPackage) throws TemplatingException {
    writeJavaFile(getJavaFileBuilderForClass(clazz, targetPackage).build());
  }

  protected JavaFile.Builder getJavaFileBuilderForClass(TypeSpec clazz, String targetPackage) {
    return JavaFile.builder(targetPackage, clazz).skipJavaLangImports(true);
  }

  protected void writeJavaFile(JavaFile javaFile) throws TemplatingException {
    try {
      javaFile.writeTo(outputDir.resolve("src/main/java"));
    } catch (Exception e) {
      throw new TemplatingException("There was an error when writing " + this.getClass().getName() + " class", e);
    }
  }

  public Path getResourcesPath() {
    return outputDir.resolve("src/main/resources");
  }

  public Path getSourcesPath() {
    return outputDir.resolve("src");
  }

  protected MethodSpec.Builder generateGetter(FieldSpec fieldSpec, CaseFormat fieldNameCaseFormat) {
    String name = "get" + fieldNameCaseFormat.to(UPPER_CAMEL, fieldSpec.name);

    return MethodSpec
        .methodBuilder(name)
        .addModifiers(Modifier.PUBLIC)
        .returns(fieldSpec.type)
        .addCode(CodeBlock.builder().addStatement("return this.$N", fieldSpec).build());
  }

  protected MethodSpec.Builder generateOptionalGetter(FieldSpec fieldSpec, Class<?> type, CaseFormat fieldNameCaseFormat) {
    String name = "get" + fieldNameCaseFormat.to(UPPER_CAMEL, fieldSpec.name);

    return MethodSpec
        .methodBuilder(name)
        .addModifiers(Modifier.PUBLIC)
        .returns(ParameterizedTypeName.get(Optional.class, type))
        .addCode(CodeBlock.builder().addStatement("return $T.ofNullable(this.$N)", Optional.class, fieldSpec).build());
  }

  protected FieldSpec getConstantStringField(String fieldName, String value) {
    FieldSpec.Builder fieldSpecBuilder = FieldSpec
        .builder(String.class, fieldName, PRIVATE, STATIC, FINAL);

    if (isNotBlank(value)) {
      fieldSpecBuilder.initializer("$S", value);
    } else {
      fieldSpecBuilder.initializer("$L", NULL);
    }

    return fieldSpecBuilder.build();
  }

  protected Class<?> getJavaType(ParameterDataType parameterDataType) {
    if (parameterDataType.equals(ParameterDataType.LOCAL_DATE_TIME)) {
      return LocalDateTime.class;
    } else if (parameterDataType.equals(ParameterDataType.ZONED_DATE_TIME)) {
      return ZonedDateTime.class;
    } else if (parameterDataType.equals(ParameterDataType.STRING)) {
      return String.class;
    } else if (parameterDataType.equals(ParameterDataType.INTEGER)) {
      return Integer.class;
    } else if (parameterDataType.equals(ParameterDataType.NUMBER)) {
      return Double.class;
    } else if (parameterDataType.equals(ParameterDataType.BOOLEAN)) {
      return Boolean.class;
    }

    throw new IllegalArgumentException(parameterDataType.getName() + " is not supported");
  }
}
