/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.templating.sdk;

import com.mulesoft.connectivity.rest.sdk.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.Parameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.templating.JavaTemplateEntity;
import com.mulesoft.connectivity.rest.commons.api.configuration.RestConfiguration;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.api.error.RequestErrorTypeProvider;
import com.mulesoft.connectivity.rest.commons.api.operation.BaseRestOperation;
import com.mulesoft.connectivity.rest.commons.api.operation.ConfigurationOverrides;
import com.mulesoft.connectivity.rest.commons.api.operation.EntityRequestParameters;
import com.mulesoft.connectivity.rest.commons.api.operation.HttpResponseAttributes;
import com.mulesoft.connectivity.rest.commons.api.operation.NonEntityRequestParameters;
import com.mulesoft.connectivity.rest.commons.internal.RestConstants;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder.QueryParamFormat;

import org.mule.runtime.extension.api.annotation.error.Throws;
import org.mule.runtime.extension.api.annotation.param.Config;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.annotation.param.MediaType;
import org.mule.runtime.extension.api.annotation.param.ParameterGroup;
import org.mule.runtime.extension.api.annotation.param.display.DisplayName;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.process.CompletionCallback;
import org.mule.runtime.extension.api.runtime.streaming.StreamingHelper;
import org.mule.runtime.http.api.HttpConstants;

import javax.lang.model.element.Modifier;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;

import static com.google.common.base.CaseFormat.LOWER_CAMEL;
import static com.google.common.base.CaseFormat.UPPER_UNDERSCORE;
import static java.util.Collections.emptyList;
import static javax.ws.rs.core.MediaType.MULTIPART_FORM_DATA_TYPE;
import static org.apache.commons.lang3.StringUtils.defaultIfEmpty;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.util.JavaUtils.getJavaLowerCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.util.JavaUtils.getJavaUpperCamelNameFromXml;
import static com.mulesoft.connectivity.rest.sdk.internal.util.JavaUtils.abbreviateText;

public abstract class AbstractSdkOperation extends JavaTemplateEntity {

  public static final String PARAM_DOC_NAME_DESCRIPTION = "@param $L $L\n";

  public static final String CONTENT_TYPE_HEADER_NAME = "content-type";
  public static final String ACCEPT_HEADER_NAME = "accept";

  public static final String ADD_QUERY_PARAM_METHOD_NAME = "addQueryParam";
  public static final String ADD_HEADER_METHOD_NAME = "addHeader";

  public static final String ADD_MULTIPLE_QUERY_PARAM_METHOD_NAME = "addQueryParams";
  public static final String ADD_MULTIPLE_HEADER_METHOD_NAME = "addHeaders";

  private static final String REQUEST_PARAMETERS_GROUP_NAME = "REQUEST_PARAMETERS_GROUP_NAME";
  private static final String CONNECTOR_OVERRIDES = "CONNECTOR_OVERRIDES";

  private static final String QUERY_PARAM_FORMAT_FIELD = "QUERY_PARAM_FORMAT";

  private final ConnectorOperation operation;
  protected final List<SdkParameter> allPathParameters;
  protected final List<SdkParameter> allQueryParameters;
  protected final List<SdkParameter> allHeaders;
  protected final SdkContent content;
  protected final SdkOutputMetadataResolver outputMetadataResolver;

  private final SdkMtfOperationTest sdkMtfOperationTest;

  public abstract FieldSpec generateExpressionLanguageField();

  public abstract ParameterSpec generateInitialPagingParameter();

  public abstract TypeName generateMethodReturn();

  public abstract CodeBlock generateOperationMethodBody();

  public AbstractSdkOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                              ConnectorOperation operation, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, runConfiguration);

    this.operation = operation;

    this.allPathParameters = buildSdkParameters(outputDir, connectorModel, sdkConnector, operation.getUriParameters());
    this.allQueryParameters = buildSdkParameters(outputDir, connectorModel, sdkConnector, operation.getQueryParameters());
    this.allHeaders = buildSdkParameters(outputDir, connectorModel, sdkConnector, operation.getHeaders());

    this.content = buildContent(outputDir, connectorModel, sdkConnector, operation);

    this.outputMetadataResolver = buildOutputMetadataResolver(sdkConnector);

    this.sdkMtfOperationTest = new SdkMtfOperationTest(operation, outputDir);
  }

  private List<SdkParameter> buildSdkParameters(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                                List<Parameter> uriParameters)
      throws TemplatingException {
    final List<SdkParameter> list = new ArrayList<>();
    for (Parameter parameter : uriParameters) {
      list.add(new SdkParameter(outputDir, connectorModel, sdkConnector, getJavaClassName(), parameter, this, runConfiguration));
    }
    return list;
  }

  private SdkOutputMetadataResolver buildOutputMetadataResolver(SdkConnector sdkConnector) throws TemplatingException {
    if (operation.getOutputMetadata() == null) {
      return null;
    }

    if (operation.hasPagination()) {
      return new SdkPagingMetadataResolver(outputDir,
                                           connectorModel,
                                           sdkConnector,
                                           operation,
                                           operation.getOutputMetadata(),
                                           runConfiguration);
    } else {
      return new SdkOutputMetadataResolver(outputDir,
                                           connectorModel,
                                           sdkConnector,
                                           operation,
                                           operation.getOutputMetadata(),
                                           runConfiguration);
    }
  }

  protected SdkContent buildContent(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                    ConnectorOperation operation)
      throws TemplatingException {
    return operation.getInputMetadata() != null
        ? new SdkContent(outputDir, connectorModel, sdkConnector, operation, runConfiguration)
        : null;
  }

  public String getJavaClassName() {
    return getJavaUpperCamelNameFromXml(operation.getInternalName()) + "Operation";
  }

  private String getJavaMethodName() {
    return getJavaLowerCamelNameFromXml(operation.getInternalName());
  }

  public String getPackage() {
    return connectorModel.getBasePackage() + ".internal.operation";
  }

  @Override
  public void applyTemplates() throws TemplatingException {
    if (content != null) {
      content.applyTemplates();
    }

    if (outputMetadataResolver != null) {
      outputMetadataResolver.applyTemplates();
    }

    for (SdkParameter sdkParameter : allPathParameters) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : allQueryParameters) {
      sdkParameter.applyTemplates();
    }

    for (SdkParameter sdkParameter : allHeaders) {
      sdkParameter.applyTemplates();
    }

    sdkMtfOperationTest.applyTemplates();

    generateOperationClass();
  }

  protected void generateOperationClass() throws TemplatingException {
    TypeSpec.Builder operationClassBuilder =
        TypeSpec
            .classBuilder(getJavaClassName())
            .addModifiers(Modifier.PUBLIC)
            .superclass(BaseRestOperation.class)
            .addMethod(generateOperationMethod());

    FieldSpec expressionLanguage = generateExpressionLanguageField();
    if (expressionLanguage != null) {
      operationClassBuilder.addField(expressionLanguage);
    }
    for (SdkParameter pathParam : this.allPathParameters) {
      operationClassBuilder.addField(generatePathParamPatternField(pathParam));
    }

    operationClassBuilder.addField(generateQueryParamFormatField());

    JavaFile.Builder javaFileBuilder = JavaFile
        .builder(getPackage(), operationClassBuilder.build())
        .skipJavaLangImports(true)
        .addStaticImport(RestConstants.class, REQUEST_PARAMETERS_GROUP_NAME, CONNECTOR_OVERRIDES);

    if (!this.allPathParameters.isEmpty()) {
      javaFileBuilder.addStaticImport(Pattern.class, "compile");
    }

    writeJavaFile(javaFileBuilder.build());
  }

  private FieldSpec generateQueryParamFormatField() {
    return FieldSpec.builder(QueryParamFormat.class,
                             QUERY_PARAM_FORMAT_FIELD,
                             Modifier.PRIVATE,
                             Modifier.FINAL,
                             Modifier.STATIC)
        .initializer("$T.$L", QueryParamFormat.class, operation.getQueryParamArrayFormat().name())
        .build();
  }

  private FieldSpec generatePathParamPatternField(SdkParameter pathParam) {
    return FieldSpec.builder(Pattern.class,
                             getPathParamPatternFieldName(pathParam),
                             Modifier.PRIVATE,
                             Modifier.FINAL,
                             Modifier.STATIC)
        .initializer("compile(\"\\\\{$L}\")", pathParam.getExternalName())
        .build();
  }

  private String getPathParamPatternFieldName(SdkParameter pathParam) {
    return LOWER_CAMEL.to(UPPER_UNDERSCORE, pathParam.getJavaName()) + "_PATTERN";
  }

  private AnnotationSpec generateDescriptionAnnotation() {
    return AnnotationSpec
        .builder(Summary.class)
        .addMember(VALUE_MEMBER, "$S", abbreviateText(operation.getDescription()))
        .build();
  }

  public MethodSpec generateOperationMethod() {
    CodeBlock.Builder javaDoc = CodeBlock.builder()
        .add("\n$L\n", defaultIfEmpty(operation.getDescription(), operation.getDisplayName()))
        .add("\nThis operation makes an HTTP $L request to the $L endpoint", operation.getHttpMethod().toUpperCase(),
             operation.getPath())
        .add("\n");

    MethodSpec.Builder methodBuilder = MethodSpec
        .methodBuilder(getJavaMethodName())
        .addModifiers(Modifier.PUBLIC);

    ParameterSpec configParameter = generateConfigParameter();
    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, configParameter.name, "the configuration to use");
    methodBuilder.addParameter(configParameter);

    if (requiresConnectionParameter()) {
      ParameterSpec connectionParameter = generateConnectionParameter();
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, connectionParameter.name, "the connection to use");
      methodBuilder.addParameter(connectionParameter);
    }

    for (SdkParameter sdkParam : allPathParameters) {
      ParameterSpec parameterSpec = sdkParam.generateParameterParameter().build();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                  defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
    }

    for (SdkParameter sdkParam : allQueryParameters) {
      if (!isQueryParamDefinedInPagination(sdkParam.getExternalName())) {
        ParameterSpec parameterSpec = sdkParam.generateParameterParameter().build();
        methodBuilder.addParameter(parameterSpec);
        javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                    defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
      }
    }

    for (SdkParameter sdkParam : allHeaders) {
      ParameterSpec parameterSpec = sdkParam.generateParameterParameter().build();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                  defaultIfEmpty(sdkParam.getDescription(), sdkParam.getDisplayName()));
    }

    ParameterSpec initialPagingParameter = generateInitialPagingParameter();
    if (initialPagingParameter != null) {
      methodBuilder.addParameter(initialPagingParameter);
      Optional<AnnotationSpec> summary =
          initialPagingParameter.annotations.stream().filter(x -> x.members.containsKey(VALUE_MEMBER)).findFirst();
      if (summary.isPresent()) {
        javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, initialPagingParameter.name,
                    summary.get().members.get(VALUE_MEMBER).get(0).toString().replaceAll("\"", ""));
      }
    }

    addContentParameters(javaDoc, methodBuilder);

    ParameterSpec parameterSpec = generateRequestParametersParameter();
    methodBuilder.addParameter(parameterSpec);

    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name,
                "the {@link " + ((ClassName) parameterSpec.type).simpleName() + "}");

    ParameterSpec configurationOverridesParameter = generateConfigurationOverridesParameter();
    methodBuilder.addParameter(configurationOverridesParameter);
    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, configurationOverridesParameter.name, "the {@link ConfigurationOverrides}");

    ParameterSpec streamingHelperParameter = generateStreamingHelperParameter();
    methodBuilder.addParameter(streamingHelperParameter);
    javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, streamingHelperParameter.name, "the {@link StreamingHelper}");

    if (requiresCallbackParameter()) {
      ParameterSpec completionCallbackParameter = generateCompletionCallbackParameter();
      methodBuilder.addParameter(completionCallbackParameter);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, completionCallbackParameter.name, "the operation's {@link CompletionCallback}");
    }

    methodBuilder.addAnnotation(generateThrowsAnnotation());

    methodBuilder.addAnnotation(generateDisplayNameAnnotation());

    if (isNotBlank(operation.getDescription())) {
      methodBuilder.addAnnotation(generateDescriptionAnnotation());
    }

    if (requiresMediaTypeAnnotation()) {
      if (operation.getOutputMetadata() != null) {
        methodBuilder.addAnnotation(generateMediaTypeAnnotation());
      } else {
        methodBuilder.addAnnotation(generateDefaultMediaTypeAnnotation());
      }
    }

    if (outputMetadataResolver != null && outputMetadataResolver.getRequiresMetadataResolver()) {
      methodBuilder.addAnnotation(outputMetadataResolver.getOutputMetadataResolverAnnotation());
    }

    methodBuilder.addCode(generateOperationMethodBody());
    methodBuilder.addJavadoc(javaDoc.build());

    TypeName returnType = generateMethodReturn();
    if (returnType != null) {
      methodBuilder.returns(returnType);
    }

    return methodBuilder.build();
  }

  protected void addContentParameters(CodeBlock.Builder javaDoc, MethodSpec.Builder methodBuilder) {
    if (content != null) {
      ParameterSpec parameterSpec = content.generateContentParameter();
      methodBuilder.addParameter(parameterSpec);
      javaDoc.add(PARAM_DOC_NAME_DESCRIPTION, parameterSpec.name, "the content to use");
    }
  }

  private boolean isQueryParamDefinedInPagination(String paramName) {
    Pagination pagination = this.connectorModel.getPagination(this.operation.getPagination());
    if (pagination != null) {
      return pagination.containsParameterExternalName(paramName);
    }

    return false;
  }

  private AnnotationSpec generateDisplayNameAnnotation() {
    return AnnotationSpec.builder(DisplayName.class)
        .addMember(VALUE_MEMBER, "$S", operation.getDisplayName())
        .build();
  }

  public CodeBlock.Builder generateCommonOperationMethodBody() {
    CodeBlock.Builder methodBody = CodeBlock.builder();

    methodBody.addStatement("$T requestPath = $S", String.class, operation.getPath());

    for (SdkParameter pathParam : allPathParameters) {
      if (pathParam.isArrayType()) {
        methodBody
            .addStatement("requestPath = $L.matcher(requestPath).replaceAll($L.stream().map(v -> $L).collect($T.joining(\",\")))",
                          getPathParamPatternFieldName(pathParam),
                          pathParam.getJavaName(),
                          pathParam.getStringValueGetter("v"),
                          Collectors.class);
      } else {
        methodBody.addStatement("requestPath = $L.matcher(requestPath).replaceAll($L)",
                                getPathParamPatternFieldName(pathParam),
                                pathParam.getStringValueGetter());
      }
    }

    String baseUriString = "connection.getBaseUri()";

    if (isNotBlank(operation.getAlternativeBaseUri()) && connectorModel.getBaseUri().isMultipleBaseUri()) {
      String multipleBaseUri =
          connectorModel.getBaseUri().getMultipleBaseUriOrDefault(operation.getAlternativeBaseUri());

      baseUriString = "\"" + multipleBaseUri + "\"";
    }

    methodBody.add("$1T builder = new $1T($2L, requestPath, $3T.$4L, parameters)",
                   RestRequestBuilder.class,
                   baseUriString,
                   HttpConstants.Method.class,
                   operation.getHttpMethod().toUpperCase());

    methodBody.add(".setQueryParamFormat($L)", QUERY_PARAM_FORMAT_FIELD);

    if (operation.getInputMetadata() != null && operation.getInputMetadata().getMediaType() != null) {

      String mediaType = operation.getInputMetadata().getMediaType().toString();

      if (operation.getInputMetadata().getMediaType().equals(MULTIPART_FORM_DATA_TYPE)) {
        mediaType = mediaType + "; boundary=__rc2_34b212";
      }

      methodBody.add(".$L($S, $S)",
                     ADD_HEADER_METHOD_NAME,
                     CONTENT_TYPE_HEADER_NAME,
                     mediaType);
    }

    if (operation.getOutputMetadata() != null && operation.getOutputMetadata().getMediaType() != null) {
      methodBody.add(".$L($S, $S)",
                     ADD_HEADER_METHOD_NAME,
                     ACCEPT_HEADER_NAME,
                     operation.getOutputMetadata().getMediaType().toString());
    }

    for (SdkParameter queryParam : allQueryParameters) {
      if (!isQueryParamDefinedInPagination(queryParam.getExternalName())) {
        methodBody.add(generateRequestBuilderParameterCodeBlock(queryParam,
                                                                ADD_QUERY_PARAM_METHOD_NAME,
                                                                ADD_MULTIPLE_QUERY_PARAM_METHOD_NAME));
      }
    }

    for (SdkParameter header : allHeaders) {
      methodBody.add(generateRequestBuilderParameterCodeBlock(header,
                                                              ADD_HEADER_METHOD_NAME,
                                                              ADD_MULTIPLE_HEADER_METHOD_NAME));
    }

    addSetBodyMethod(methodBody);

    methodBody.add(";");

    return methodBody;
  }

  protected void addSetBodyMethod(CodeBlock.Builder methodBody) {
    if (content != null) {
      methodBody.add(".setBody($L, overrides.getStreamingType())", content.getContentParameterJavaName());
    }
  }

  private CodeBlock generateRequestBuilderParameterCodeBlock(SdkParameter parameter, String addSingleValueMethodName,
                                                             String addMultipleValueMethodName) {
    CodeBlock.Builder builder = CodeBlock.builder();

    String methodName = parameter.isArrayType() ? addMultipleValueMethodName : addSingleValueMethodName;

    builder.add(".$L($S, $L)", methodName, parameter.getExternalName(), getParameterValueStatement(parameter));

    return builder.build();
  }

  private CodeBlock getParameterValueStatement(SdkParameter parameter) {
    CodeBlock.Builder builder = CodeBlock.builder();

    if (parameter.isArrayType()) {
      String getter = parameter.getInnerTypeStringValueGetter("v");
      if (getter.equals("v")) {
        builder.add("$L.stream().filter($T::nonNull).collect($T.toList())",
                    parameter.getJavaName(),
                    Objects.class,
                    Collectors.class);
      } else {
        builder.add("$L.stream().filter($T::nonNull).map(v -> $L).collect($T.toList())",
                    parameter.getJavaName(),
                    Objects.class,
                    getter,
                    Collectors.class);
      }
    } else {
      builder.add("$L", parameter.getStringValueGetter());
    }

    return builder.build();
  }

  private AnnotationSpec generateThrowsAnnotation() {
    return AnnotationSpec
        .builder(Throws.class)
        .addMember(VALUE_MEMBER, "$T.class", RequestErrorTypeProvider.class)
        .build();
  }

  private AnnotationSpec generateMediaTypeAnnotation() {
    return AnnotationSpec
        .builder(MediaType.class)
        .addMember(VALUE_MEMBER, "$S", operation.getOutputMetadata().getMediaType())
        .build();
  }

  private AnnotationSpec generateDefaultMediaTypeAnnotation() {
    return AnnotationSpec
        .builder(MediaType.class)
        .addMember(VALUE_MEMBER, "$S", operationMethodRequiresBody() ? "application/json" : "text/plain")
        .build();
  }

  private ParameterSpec generateConfigParameter() {
    return ParameterSpec
        .builder(RestConfiguration.class, "config")
        .addAnnotation(Config.class)
        .build();
  }

  private ParameterSpec generateConnectionParameter() {
    return ParameterSpec
        .builder(RestConnection.class, "connection")
        .addAnnotation(Connection.class)
        .build();
  }

  private ParameterSpec generateRequestParametersParameter() {
    AnnotationSpec parameterGroupAnnotation =
        AnnotationSpec
            .builder(ParameterGroup.class)
            .addMember(NAME_MEMBER, REQUEST_PARAMETERS_GROUP_NAME)
            .build();

    Class<?> requestParameterClass =
        operation.getInputMetadata() != null ? EntityRequestParameters.class : NonEntityRequestParameters.class;

    return ParameterSpec
        .builder(requestParameterClass, "parameters")
        .addAnnotation(parameterGroupAnnotation)
        .build();
  }

  private ParameterSpec generateConfigurationOverridesParameter() {
    AnnotationSpec parameterGroupAnnotation =
        AnnotationSpec
            .builder(ParameterGroup.class)
            .addMember(NAME_MEMBER, CONNECTOR_OVERRIDES)
            .build();

    return ParameterSpec
        .builder(ConfigurationOverrides.class, "overrides")
        .addAnnotation(parameterGroupAnnotation)
        .build();
  }

  private ParameterSpec generateStreamingHelperParameter() {
    return ParameterSpec
        .builder(StreamingHelper.class, "streamingHelper")
        .build();
  }

  private ParameterSpec generateCompletionCallbackParameter() {
    if (!isVoidOperation() && (outputMetadataResolver != null || operationMethodRequiresBody())) {
      return ParameterSpec
          .builder(ParameterizedTypeName.get(CompletionCallback.class, InputStream.class, HttpResponseAttributes.class),
                   "callback")
          .build();
    } else {
      return ParameterSpec
          .builder(ParameterizedTypeName.get(CompletionCallback.class, String.class, HttpResponseAttributes.class), "callback")
          .build();
    }
  }

  protected boolean operationMethodRequiresBody() {
    return operation.getHttpMethod().equalsIgnoreCase("get")
        || operation.getHttpMethod().equalsIgnoreCase("post")
        || operation.getHttpMethod().equalsIgnoreCase("patch")
        || operation.getHttpMethod().equalsIgnoreCase("options");
  }

  protected boolean isVoidOperation() {
    return operation.getVoidOperation() != null && operation.getVoidOperation();
  }

  protected boolean requiresConnectionParameter() {
    return true;
  }

  protected boolean requiresCallbackParameter() {
    return true;
  }

  protected boolean requiresMediaTypeAnnotation() {
    return true;
  }

  public SdkParameter getSdkParameter(ParameterType parameterType, String parameterName) {
    List<SdkParameter> parameters = emptyList();

    if (parameterType.equals(ParameterType.URI)) {
      parameters = allPathParameters;
    } else if (parameterType.equals(ParameterType.QUERY)) {
      parameters = allQueryParameters;
    } else if (parameterType.equals(ParameterType.HEADER)) {
      parameters = allHeaders;
    }

    return parameters.stream().filter(x -> x.getExternalName().equalsIgnoreCase(parameterName)).findFirst().orElse(null);
  }
}
