/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.templating.sdk;

import com.mulesoft.connectivity.rest.sdk.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.InPaginationParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.OutPaginationParameter;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.Pagination;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.pagination.PaginationParameter;
import com.mulesoft.connectivity.rest.commons.api.connection.RestConnection;
import com.mulesoft.connectivity.rest.commons.internal.util.RestRequestBuilder;

import org.mule.runtime.api.el.ExpressionLanguage;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.extension.api.annotation.param.Optional;
import org.mule.runtime.extension.api.annotation.param.display.Summary;
import org.mule.runtime.extension.api.runtime.streaming.PagingProvider;

import javax.inject.Inject;
import javax.lang.model.element.Modifier;
import java.nio.file.Path;
import java.util.function.Function;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.ParameterSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;

import static com.mulesoft.connectivity.rest.sdk.internal.util.JavaUtils.abbreviateText;


public abstract class AbstractSdkPaginationOperation extends AbstractSdkOperation implements SdkPaginationStrategy {

  public static final String PAGING_RESPONSE_EXPRESSION = "pagingResponseExpression";

  private final Pagination pagination;

  public abstract String getInitialPagingParameterSummary();

  public AbstractSdkPaginationOperation(Path outputDir, ConnectorModel connectorModel, SdkConnector sdkConnector,
                                        ConnectorOperation operation, RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, operation, runConfiguration);
    this.pagination = connectorModel.getPagination(operation.getPagination());
  }

  public Pagination getPagination() {
    return pagination;
  }

  public AnnotationSpec buildSummaryAnnotation() {
    return AnnotationSpec
        .builder(Summary.class)
        .addMember(VALUE_MEMBER, "$S", abbreviateText(getInitialPagingParameterSummary()))
        .build();
  }

  public AnnotationSpec buildDefaultAnnotation() {
    return AnnotationSpec.builder(Optional.class)
        .addMember("defaultValue", "$S", getInitialValue())
        .build();
  }

  @Override
  public ParameterSpec generateInitialPagingParameter() {
    return ParameterSpec
        .builder(TypeName.INT, getInitialParamName())
        .addAnnotation(buildDefaultAnnotation())
        .addAnnotation(buildSummaryAnnotation())
        .build();
  }

  @Override
  public CodeBlock getPagingMethodOperation() {
    return CodeBlock.builder()
        .addStatement(
                      "return new $T($S, $L, requestFactory, expressionLanguage, streamingHelper, $S, resolveDefaultResponseMediaType(config), overrides.getResponseTimeoutAsMillis())",
                      getPagingProvider(),
                      getTokenParamName(),
                      generateInitialPagingParameter().name,
                      getPayloadExpression())
        .build();
  }

  @Override
  public TypeName generateMethodReturn() {
    return ParameterizedTypeName.get(ClassName.get(PagingProvider.class),
                                     TypeName.get(RestConnection.class),
                                     ParameterizedTypeName.get(TypedValue.class, String.class));
  }

  @Override
  public CodeBlock generateOperationMethodBody() {
    CodeBlock.Builder paginationBody = CodeBlock.builder();

    CodeBlock.Builder requestMethodBodyBuilder = super.generateCommonOperationMethodBody();

    requestMethodBodyBuilder.addStatement("return builder");
    paginationBody.add("$T<$T,$T> requestFactory = connection -> { $L }; ",
                       Function.class,
                       RestConnection.class,
                       RestRequestBuilder.class,
                       requestMethodBodyBuilder.build());

    paginationBody.add(getPagingMethodOperation());

    return paginationBody.build();
  }

  @Override
  public FieldSpec generateExpressionLanguageField() {
    return FieldSpec
        .builder(ExpressionLanguage.class, "expressionLanguage", Modifier.PRIVATE)
        .addAnnotation(AnnotationSpec.builder(Inject.class).build())
        .build();
  }

  @Override
  public String getInitialValue() {
    PaginationParameter parameter = getPagination().getParameter(getInitialParamName());
    if (parameter == null) {
      return "0";
    }

    return (parameter instanceof InPaginationParameter)
        ? ((InPaginationParameter) parameter).getValueExtraction().getContent()
        : ((OutPaginationParameter) parameter).getValue();
  }

  @Override
  public String getPayloadExpression() {
    PaginationParameter parameter = this.pagination.getParameter(PAGING_RESPONSE_EXPRESSION);

    return (parameter == null) ? null : ((InPaginationParameter) parameter).getValueExtraction().getContent();
  }

  @Override
  protected boolean requiresConnectionParameter() {
    return false;
  }

  @Override
  protected boolean requiresCallbackParameter() {
    return false;
  }

  @Override
  protected boolean requiresMediaTypeAnnotation() {
    return false;
  }

}
