/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.templating.sdk;

import static javax.ws.rs.core.MediaType.APPLICATION_JSON_TYPE;
import static javax.ws.rs.core.MediaType.APPLICATION_XML_TYPE;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.util.FileGenerationUtils.SchemaNameType.OUTPUT;

import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.dynamic.FromJsonOutputDynamicMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.dynamic.JsonOutputDynamicMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.dynamic.XmlOutputDynamicMetadataResolver;
import com.mulesoft.connectivity.rest.sdk.api.RestSdkRunConfiguration;
import com.mulesoft.connectivity.rest.sdk.exception.TemplatingException;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.PrimitiveTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.TypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.CustomTypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.JsonTypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.TypeSchema;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.schema.XmlTypeSchema;

import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.FromJsonOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.JsonOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.StringOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.BooleanOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.DateOnlyOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.DateTimeOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.FileOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.IntegerOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.NumberOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.TimeOnlyOutputMetadataResolver;
import com.mulesoft.connectivity.rest.commons.api.datasense.resolver.output.XmlOutputMetadataResolver;

import org.mule.runtime.extension.api.annotation.metadata.OutputResolver;

import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;

import java.nio.file.Path;

public class SdkOutputMetadataResolver extends SdkAbstractStaticMetadataResolver {

  private String qName;

  public SdkOutputMetadataResolver(Path outputDir,
                                   ConnectorModel connectorModel,
                                   SdkConnector sdkConnector,
                                   ConnectorOperation operation,
                                   TypeDefinition typeDefinition,
                                   RestSdkRunConfiguration runConfiguration)
      throws TemplatingException {
    super(outputDir, connectorModel, sdkConnector, operation, typeDefinition, OUTPUT, EMPTY, runConfiguration);
  }

  protected Class<?> buildSuperclass(TypeDefinition typeDefinition) throws TemplatingException {
    TypeSchema typeSchema = typeDefinition.getTypeSchema();

    if (typeDefinition instanceof PrimitiveTypeDefinition) {
      return getPrimitiveMetadataResolverClass((PrimitiveTypeDefinition) typeDefinition);

    } else if (typeSchema instanceof XmlTypeSchema || typeDefinition.getMediaType().isCompatible(APPLICATION_XML_TYPE)) {
      qName = ((XmlTypeSchema) typeDefinition.getTypeSchema()).getElementName();
      return runConfiguration.useDynamicTypeResolvers() ? XmlOutputDynamicMetadataResolver.class
          : XmlOutputMetadataResolver.class;

    } else if (typeSchema instanceof JsonTypeSchema || typeDefinition.getMediaType().isCompatible(APPLICATION_JSON_TYPE)) {
      return runConfiguration.useDynamicTypeResolvers() ? JsonOutputDynamicMetadataResolver.class
          : JsonOutputMetadataResolver.class;

    } else if (typeSchema instanceof CustomTypeSchema) {
      return runConfiguration.useDynamicTypeResolvers() ? FromJsonOutputDynamicMetadataResolver.class
          : FromJsonOutputMetadataResolver.class;
    }

    return StringOutputMetadataResolver.class;
  }

  protected Class<?> getPrimitiveMetadataResolverClass(PrimitiveTypeDefinition primitiveTypeDefinition)
      throws TemplatingException {
    PrimitiveTypeDefinition.PrimitiveType primitiveType = primitiveTypeDefinition.getPrimitiveType();

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.NUMBER)) {
      return NumberOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.INTEGER)) {
      return IntegerOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.STRING)) {
      return StringOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.BOOLEAN)) {
      return BooleanOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.DATE_TIME)
        || primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.DATE_TIME_ONLY)) {
      return DateTimeOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.TIME_ONLY)) {
      return TimeOnlyOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.DATE_ONLY)
        || primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.DATE)) {
      return DateOnlyOutputMetadataResolver.class;
    }

    if (primitiveType.equals(PrimitiveTypeDefinition.PrimitiveType.FILE)) {
      return FileOutputMetadataResolver.class;
    }

    throw new TemplatingException("Primitive type not supported: " + primitiveType.name());
  }

  public AnnotationSpec getOutputMetadataResolverAnnotation() {
    return AnnotationSpec
        .builder(OutputResolver.class)
        .addMember("output", "$T.class", ClassName.get(getPackage(), getClassName()))
        .build();
  }

  @Override
  protected String getClassNameSuffix() {
    return "OutputMetadataResolver";
  }

  @Override
  protected boolean requiresQNameMethod() {
    return superclass.equals(XmlOutputMetadataResolver.class)
        || superclass.equals(XmlOutputDynamicMetadataResolver.class);
  }

  @Override
  protected boolean requiresFormatMethod() {
    return superclass.equals(FromJsonOutputMetadataResolver.class)
        || superclass.equals(FileOutputMetadataResolver.class)
        || superclass.equals(FromJsonOutputDynamicMetadataResolver.class);
  }

  @Override
  protected String getQName() {
    return qName;
  }

  @Override
  protected boolean requiresToWriteSchema() {
    return superclass.equals(FromJsonOutputMetadataResolver.class)
        || superclass.equals(JsonOutputMetadataResolver.class)
        || superclass.equals(XmlOutputMetadataResolver.class)
        || superclass.equals(FromJsonOutputDynamicMetadataResolver.class)
        || superclass.equals(JsonOutputDynamicMetadataResolver.class)
        || superclass.equals(XmlOutputDynamicMetadataResolver.class);
  }

}
