/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.parameter.ParameterType;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ParameterDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIParameterModel;

import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

public class DescriptorParameterMustBePresentInApiRule extends PreValidationRule {

  //R016
  public DescriptorParameterMustBePresentInApiRule() {
    super("Operation parameters declared in the connector descriptor must be present in the API spec",
          EMPTY,
          ERROR);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    List<ValidationResult> results = new LinkedList<>();

    for (EndPointDescriptor endPointDescriptor : connectorDescriptor.getEndpoints()) {
      for (OperationDescriptor operationDescriptor : endPointDescriptor.getOperations()) {
        if (operationDescriptor.getExpects() != null) {
          results.addAll(validateParametersExists(operationDescriptor.getExpects().getUriParameter(), ParameterType.URI,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

          results.addAll(validateParametersExists(operationDescriptor.getExpects().getQueryParameter(), ParameterType.QUERY,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

          results.addAll(validateParametersExists(operationDescriptor.getExpects().getHeader(), ParameterType.HEADER,
                                                  endPointDescriptor.getPath(), operationDescriptor.getMethod(), apiModel));

        }
      }
    }

    return results;
  }

  private List<ValidationResult> validateParametersExists(List<ParameterDescriptor> parameters,
                                                          ParameterType parameterType,
                                                          String path,
                                                          String method,
                                                          APIModel apiModel) {
    return parameters.stream()
        .map(x -> validateParameterExists(x, parameterType, path, method, apiModel))
        .filter(Objects::nonNull)
        .collect(toList());
  }

  private ValidationResult validateParameterExists(ParameterDescriptor parameter,
                                                   ParameterType parameterType,
                                                   String path,
                                                   String method,
                                                   APIModel apiModel) {

    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path))
        .filter(x -> x.getHttpMethod().name().equalsIgnoreCase(method))
        .findFirst()
        .map(x -> validateParameterExists(x, parameter, parameterType))
        .orElse(null);
  }

  private ValidationResult validateParameterExists(APIOperationModel apiOperation, ParameterDescriptor parameter,
                                                   ParameterType parameterType) {
    List<APIParameterModel> apiParameterList;

    if (parameterType.equals(ParameterType.URI)) {
      apiParameterList = apiOperation.getUriParamsModel();
    } else if (parameterType.equals(ParameterType.QUERY)) {
      apiParameterList = apiOperation.getQueryParamsModel();
    } else {
      apiParameterList = apiOperation.getHeadersModel();
    }

    return apiParameterList.stream().anyMatch(x -> x.getExternalName().equalsIgnoreCase(parameter.getParamName()))
        ? null
        : getValidationError(apiOperation, parameter, parameterType);
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel, ParameterDescriptor parameter,
                                              ParameterType parameterType) {
    String location =
        "Operation with PATH: " + apiOperationModel.getPath() +
            " and METHOD: " + apiOperationModel.getHttpMethod().name() +
            " does not declare a " + parameterType.getName() + " named " + parameter.getParamName();

    return new ValidationResult(this, location);
  }
}
