/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static java.util.Collections.emptyList;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.EndPointDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.LinkedList;
import java.util.List;

public class NoInputMediaTypeDefinedRule extends PreValidationRule {

  //R017
  public NoInputMediaTypeDefinedRule() {
    super("Defining a media type for the operations input type is mandatory",
          "If an operation defines an input type it should also define a media type. This can be done at the operation level or globally.",
          ERROR);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    if (isNotBlank(connectorDescriptor.getDefaultInputMediaType())) {
      return emptyList();
    }

    List<ValidationResult> allErrors = new LinkedList<>();

    apiModel.getOperationsModel().stream()
        .filter(x -> !x.getInputMetadataModel().isEmpty())
        .filter(x -> !inputMediaTypeDefinedInAPISpec(x))
        .filter(x -> !inputMediaTypeDefinedInDescriptor(x, connectorDescriptor))
        .map(this::getValidationErrorForApiSpec)
        .forEach(allErrors::add);

    for (EndPointDescriptor endpoint : connectorDescriptor.getEndpoints()) {
      endpoint.getOperations().stream()
          .filter(x -> isNotBlank(x.getInputTypeSchema()))
          .filter(x -> isBlank(x.getInputMediaType()))
          .filter(x -> !inputMediaTypeDefinedInAPISpec(apiModel, endpoint.getPath(), x.getMethod()))
          .map(x -> getValidationErrorForDescriptor(endpoint.getPath(), x.getMethod()))
          .forEach(allErrors::add);
    }

    return allErrors;
  }

  private boolean inputMediaTypeDefinedInAPISpec(APIOperationModel apiOperation) {
    return apiOperation.getInputMetadataModel().stream()
        .anyMatch(x -> x.getMediaType() != null);
  }

  private boolean inputMediaTypeDefinedInAPISpec(APIModel apiModel, String path, String method) {
    return apiModel.getOperationsModel().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(path) && x.getHttpMethod().name().equalsIgnoreCase(method))
        .flatMap(x -> x.getInputMetadataModel().stream())
        .anyMatch(x -> x.getMediaType() != null);
  }

  private boolean inputMediaTypeDefinedInDescriptor(APIOperationModel apiOperationModel, ConnectorDescriptor descriptor) {
    OperationDescriptor operationDescriptor = descriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .findFirst().orElse(null);

    return operationDescriptor != null && isNotBlank(operationDescriptor.getInputMediaType());
  }

  private ValidationResult getValidationErrorForApiSpec(APIOperationModel apiOperationModel) {
    String location =
        "Operation with PATH: " + apiOperationModel.getPath() +
            " and METHOD: " + apiOperationModel.getHttpMethod().name() + " defined in the API spec";
    return new ValidationResult(this, location);
  }

  private ValidationResult getValidationErrorForDescriptor(String path, String method) {
    String location =
        "Operation with PATH: " + path +
            " and METHOD: " + method + " defined in the Connector Descriptor";
    return new ValidationResult(this, location);
  }
}
