/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.EMPTY;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.WARN;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.security.NotSupportedScheme;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PreValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.List;

public class NotSupportedSecuritySchemeRule extends PreValidationRule {

  //R010
  public NotSupportedSecuritySchemeRule() {
    super("The API spec defines a security scheme that is not supported",
          EMPTY,
          WARN);
  }

  @Override
  public List<ValidationResult> preValidate(ConnectorDescriptor connectorDescriptor, APIModel apiModel) {

    return apiModel.getOperationsModel().stream()
        .filter(this::anySecuritySchemeNotSupported)
        .map(this::getValidationError)
        .collect(toList());
  }

  private boolean anySecuritySchemeNotSupported(APIOperationModel apiOperationModel) {
    return !apiOperationModel.getSecuritySchemesModel().isEmpty() &&
        apiOperationModel.getSecuritySchemesModel().stream()
            .anyMatch(x -> x.getSecuritySchemeClass().equals(NotSupportedScheme.class));
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel) {
    String location =
        "Operation with PATH: " + apiOperationModel.getPath() + " and METHOD: " + apiOperationModel.getHttpMethod().name();

    return new ValidationResult(this, location);
  }
}
