/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.connectivity.rest.sdk.internal.validation.rules;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationRule.Level.ERROR;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorModel;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.ConnectorOperation;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.HTTPMethod;
import com.mulesoft.connectivity.rest.sdk.internal.connectormodel.type.EmptyTypeDefinition;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.ConnectorDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.descriptor.model.OperationDescriptor;
import com.mulesoft.connectivity.rest.sdk.internal.validation.PostValidationRule;
import com.mulesoft.connectivity.rest.sdk.internal.validation.ValidationResult;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIModel;
import com.mulesoft.connectivity.rest.sdk.internal.webapi.model.APIOperationModel;

import java.util.List;

public class OperationMustDeclareResponseBodyRule extends PostValidationRule {

  //R003
  public OperationMustDeclareResponseBodyRule() {
    super("Operations must declare a response body",
          "GET, POST, PATCH and OPTIONS operations defined in the API spec must declare a response body and a type for that body. If they don't the connector descriptor must skip this check, define a response body for the operation, or declare it as void.",
          ERROR);
  }

  @Override
  public List<ValidationResult> postValidate(ConnectorDescriptor connectorDescriptor,
                                             APIModel apiModel,
                                             ConnectorModel connectorModel) {

    List<APIOperationModel> nullOutputAPIOperations =
        connectorModel.getOperations().stream()
            .filter(x -> httpMethodApplies(x.getHttpMethod()))
            .filter(x -> x.getOutputMetadata() == null || x.getOutputMetadata() instanceof EmptyTypeDefinition)
            .map(x -> getApiOperationModel(apiModel, x))
            .collect(toList());

    return nullOutputAPIOperations.stream()
        .filter(nullOutputOperation -> !validationSkipped(connectorModel, nullOutputOperation))
        .filter(nullOutputOperation -> !isVoidOperation(connectorDescriptor, nullOutputOperation))
        .map(this::getValidationError)
        .collect(toList());
  }

  private APIOperationModel getApiOperationModel(APIModel apiModel, ConnectorOperation connectorOperation) {
    final APIOperationModel apiOperationModel = apiModel.getOperationsModel().stream()
        .filter(x -> connectorOperation.getPath().equalsIgnoreCase(x.getPath()))
        .filter(x -> connectorOperation.getHttpMethod().equalsIgnoreCase(x.getHttpMethod().name()))
        .findFirst().orElse(null);

    return requireNonNull(apiOperationModel);
  }

  private boolean isVoidOperation(ConnectorDescriptor connectorDescriptor, APIOperationModel apiOperationModel) {
    OperationDescriptor operationDescriptor = connectorDescriptor.getEndpoints().stream()
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .flatMap(x -> x.getOperations().stream())
        .filter(x -> x.getMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .findFirst().orElse(null);

    return operationDescriptor != null && operationDescriptor.getVoidOperation() != null
        && operationDescriptor.getVoidOperation();
  }

  private boolean httpMethodApplies(String httpMethod) {
    return httpMethod.equalsIgnoreCase(HTTPMethod.GET.name())
        || httpMethod.equalsIgnoreCase(HTTPMethod.POST.name())
        || httpMethod.equalsIgnoreCase(HTTPMethod.PATCH.name())
        || httpMethod.equalsIgnoreCase(HTTPMethod.OPTIONS.name());
  }

  private boolean validationSkipped(ConnectorModel connectorModel, APIOperationModel apiOperationModel) {
    boolean globalSkip = connectorModel.getSkipOutputTypeValidation() != null && connectorModel.getSkipOutputTypeValidation();

    ConnectorOperation connectorOperation = connectorModel.getOperations().stream()
        .filter(x -> x.getHttpMethod().equalsIgnoreCase(apiOperationModel.getHttpMethod().name()))
        .filter(x -> x.getPath().equalsIgnoreCase(apiOperationModel.getPath()))
        .findFirst().orElse(null);

    if (connectorOperation == null) {
      return false;
    }

    if (connectorOperation.getSkipOutputTypeValidation() == null) {
      return globalSkip;
    } else {
      return connectorOperation.getSkipOutputTypeValidation();
    }
  }

  private ValidationResult getValidationError(APIOperationModel apiOperationModel) {
    String location =
        "Operation with PATH: " + apiOperationModel.getPath() + " and METHOD: " + apiOperationModel.getHttpMethod().name();
    return new ValidationResult(this, location);
  }
}
